/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.TreeMap;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;

/**
 * Class for parsing standard data types and arrays in and out of an XML
 * {@link java.io.File}. The methods with prefix <code>append</code> or
 * <code>add</code> are for encoding, while methods with prefix
 * <code>extract</code> are for decoding.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public class XMLParser {

	/**
	 * Parses the XML-attributes given in <code>attrs</code> and returns a Map of attributed names and associated values (as {@link String}s).
	 * @param attrs the list of XML-attributes
	 * @return the {@link Map} of attribute names and values
	 * @throws NonParsableException if any of the attributes does not have a value
	 */
	private static Map<String, String> parseAttributes( String attrs ) throws NonParsableException {
		Map<String, String> map = new TreeMap<String, String>();
		String[] parts = attrs.split( "(?<!=)\\s+(?!=)" );
		int vallength = 0;
		for( String part : parts ) {
			part = part.trim();
			if( part.length() > 0 ) {
				String[] keyVal = part.split( "=" );
				if( keyVal.length != 2 ) {
					throw new NonParsableException( "Malformed attributes: " + attrs );
				}
				keyVal[0] = keyVal[0].trim();
				keyVal[1] = keyVal[1].trim();
				vallength = keyVal[1].length();
				if( keyVal[1].charAt( 0 ) == '"' && keyVal[1].charAt( vallength - 1 ) == '"' ) {
					keyVal[1] = keyVal[1].substring( 1, vallength - 1 );
				}
				map.put( keyVal[0], keyVal[1] );
			}
		}
		return map;
	}

	/**
	 * Tests whether the attributes given in <code>filterAttributes</code> are present in <code>myAttrs</code> and if the values of present attributes are equal.
	 * @param myAttrs the attributes to be tested
	 * @param filterAttributes the attributes and associated values that must be present in <code>myAttrs</code>
	 * @return true if all attributes and values in <code>filterAttributes</code> could be found in <code>myAttrs</code>, false otherwise
	 */
	private static boolean testFilter( Map<String, String> myAttrs, Map<String, String> filterAttributes ) {

		Set<String> keys = filterAttributes.keySet();
		String myVal = null, filterVal = null;

		for( String key : keys ) {
			myVal = myAttrs.get( key );
			filterVal = filterAttributes.get( key );
			if( myVal != null || filterVal != null ) {
				if( myVal == null ) {
					return false;
				}
				myVal = myVal.trim();
				filterVal = filterVal.trim();
				if( !myVal.equals( filterVal ) ) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Extracts the contents of <code>source</code> between <code>tag</code> start and end tags.
	 * @param source the XML-code containing start and end tag
	 * @param tag the tag (without angle brackets)
	 * @return the contents of start and end tags, without these tags, as a {@link StringBuffer}
	 * @throws NonParsableException if start or end tag could not be found
	 */
	public static StringBuffer extractForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractForTag( source, tag, null, null );
	}
	
	/**
	 * Extracts the contents of <code>source</code> between <code>tag</code> start and end tags.
	 * If <code>attributes</code> is not <code>null</code>, the attributes of the start tag are added to this {@link Map}.
	 * If <code>filterAttributes</code> is not <code>null</code>, the start tag is accepted only if its attributes and associated values contain those defined in <code>filterAttributed</code>.
	 * @param source the XML-code containing start and end tag
	 * @param tag the tag (without angle brackets)
	 * @param attributes a {@link Map} for attributes and values, or <code>null</code> if no attributes should be parsed.
	 * @param filterAttributes a {@link Map} of attributes and associated values, which must be present in the attributes of the start tag, or <code>null</code> for no filtering
	 * @return the contents of start and end tags, without these tags, as a {@link StringBuffer}
	 * @throws NonParsableException if start or end tag could not be found
	 */
	public static StringBuffer extractForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		if( source == null || source.length() == 0 ) {
			return null;
		}
		
		//find position of first match tag (& attributes)
		int taglength = tag.length(), start, endOfStart = 0;
		Map<String, String> myMap = null;
		boolean found = false;
		char c;
		do{
			start = source.indexOf( "<" + tag, endOfStart );
			if(start < 0){
	 			return null;
			}
			endOfStart = source.indexOf( ">", start + taglength );
			if( start < 0 || endOfStart < 0 ) {
				throw new NonParsableException( "Could not find appropriate start tag for tag " + tag
												+ " and attributes "
												+ Arrays.toString( filterAttributes.keySet().toArray() ) );
			}
			endOfStart += 1;
			c = source.charAt( start+taglength+1 );
			if( c == ' ' || c == '>' ) {
				if( filterAttributes != null || attributes != null ) {
					myMap = parseAttributes( source.substring( start + taglength + 2, endOfStart - 1 ) );
				}
				if( filterAttributes != null ) {
					found = testFilter( myMap, filterAttributes );
				} else  {
					found = true;
				}
			}
		} while( !found );
		Stack<String> tags = new Stack<String>();
		tags.push( tag );

		//find end tag
		int pos = 0, endOfTag;
		int closepos = 0;
		int closepos2 = 0;
		int nextpos = source.indexOf( "<", endOfStart );
		String current;
		while( ( pos = nextpos ) > 0 ) {
			closepos = source.indexOf( ">", pos + 1 );
			nextpos = source.indexOf( "<", pos + 1 );

			if( source.charAt( pos - 1 ) == '\\' ) {
				//< is escaped -> not XML
				continue;
			}
			if( nextpos >= 0 && nextpos < closepos ) {
				//another < before >, no < within a tag, hence, skip
				continue;
			}

			if( source.charAt( pos + 1 ) == '/' ) {
				if( source.substring( pos + 2, closepos ).equals( tags.peek() ) ) {
					tags.pop();
				} else {
					throw new NonParsableException( "Malformed XML: Nested tags may not overlap. Open tags: " + tags.toString()
													+ ",  found end tag "
													+ source.substring( pos + 2, closepos ) );
				}
			} else {
				closepos2 = source.indexOf( " ", pos + 1 );
				if( closepos2 < 0 ) {
					closepos2 = closepos;
				}
				endOfTag = Math.min( closepos, closepos2 );
				current = source.substring( pos + 1, endOfTag );
				if( current.startsWith( "!--" ) //comment
						|| current.startsWith( "![CDATA[" ) //CDATA
						|| current.startsWith( "?" ) //processing information
						|| source.charAt( closepos-1 ) == '/' //empty element tag 
					) {
					continue;
				} else {
					tags.push( source.substring( pos + 1, endOfTag ) );
				}
			}
			if( tags.size() == 0 ) {
				if( attributes != null ) {
					attributes.putAll( myMap );
				}
				StringBuffer result = new StringBuffer( source.substring( endOfStart, pos ) );
				source.delete( start, closepos+1 ); //same as: pos+taglength+3 );
				return result;
			}
		}
		throw new NonParsableException( "Malformed XML: No end tag found for " + tag + "." );
	}

	/**
	 * Returns the value between the tags as <code>boolean</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as <code>boolean</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractBooleanAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static boolean extractBooleanForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractBooleanAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>boolean</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as <code>boolean</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static boolean extractBooleanAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		return Boolean.valueOf( extractForTag( source, tag, attributes, filterAttributes ).toString().trim() ).booleanValue();
	}

	/**
	 * Returns the value between the tags as <code>byte</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as <code>byte</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractByteAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static byte extractByteForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractByteAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>byte</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as <code>byte</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static byte extractByteAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		try {
			return Byte.parseByte( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + tag + "\" to int." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
	}

	/**
	 * Returns the value between the tags as {@link Enum}.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the enumeration between the tags as {@link Enum}
	 * 
	 * @throws NonParsableException
	 *             if the enumeration could not be parsed
	 * 
	 * @see XMLParser#extractEnumAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static <T extends Enum<T>> T extractEnumForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractEnumAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as {@link Enum}.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the enumeration between the tags as {@link Enum}
	 * 
	 * @throws NonParsableException
	 *             if the enumeration could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	@SuppressWarnings( "unchecked" )
	public static <T extends Enum<T>> T extractEnumAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
		String enumName = XMLParser.extractStringForTag( ex, "enumName" );
		T erg;
		try {
			erg = Enum.valueOf( (Class<T>)Class.forName( enumName ), XMLParser.extractStringForTag( ex, "name" ) );
		} catch ( Exception e ) {
			throw getNonParsableException( "problem at " + enumName + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(),
					e );
		}
		return erg;
	}

	/**
	 * Returns the value between the tags as <code>int</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as <code>int</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractIntAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static int extractIntForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractIntAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>int</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as <code>int</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static int extractIntAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		try {
			return Integer.parseInt( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + tag + "\" to int." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
	}

	/**
	 * Returns the value between the tags as <code>long</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as <code>long</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractLongAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static long extractLongForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractLongAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>long</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as <code>long</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static long extractLongAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		try {
			return Long.parseLong( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + tag + "\" to int." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
	}

	/**
	 * Returns the value between the tags as <code>double</code>
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as <code>double</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractDoubleAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static double extractDoubleForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractDoubleAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>double</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as <code>double</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static double extractDoubleAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		try {
			return Double.parseDouble( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + tag + "\" to double." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
	}

	/**
	 * Returns the value between the tags as {@link Storable} or <code>null</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as {@link Storable}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable extractStorableForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorableAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as {@link Storable} or <code>null</code>. The concrete class of the {@link Storable}
	 * is inferred from the XML.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as {@link Storable}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable extractStorableAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		return extractStorableAndAttributesForTag( source, tag, attributes, filterAttributes, null );
	}
	
	/**
	 * Returns the value between the tags as {@link Storable} or <code>null</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param clazz
	 * 			  the concrete class of the {@link Storable}, can be <code>null</code> then the class is inferred from the XML,
	 * 			  otherwise it is tried to create an instance of the class without checking whether it is possible. This might
	 * 			  result in an {@link NonParsableException}
	 * 
	 * @return the value between the tags as {@link Storable}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable extractStorableForTag( StringBuffer source, String tag, Class<? extends Storable> clazz ) throws NonParsableException {
		return extractStorableAndAttributesForTag( source, tag, null, null, clazz );
	}
	
	/**
	 * Returns the value between the tags as {@link Storable} or <code>null</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * @param clazz
	 * 			  the concrete class of the {@link Storable}, can be <code>null</code> then the class is inferred from the XML,
	 * 			  otherwise it is tried to create an instance of the class without checking whether it is possible. This might
	 * 			  result in an {@link NonParsableException}
	 * 
	 * @return the value between the tags as {@link Storable}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static Storable extractStorableAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes, Class<? extends Storable> clazz ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
		if( ex.toString().trim().equals( "null" ) ) {
			return null;
		}
			
		if( clazz == null ) {
			try {
				clazz = (Class<? extends Storable>) Class.forName( XMLParser.extractStringForTag( ex, "className" ) );
			}catch( Exception e ) {
				throw getNonParsableException( "Class not found.", e );
			}
		}
		
		Storable erg;
		try {
			erg = (Storable)clazz.getConstructor( new Class[]{ StringBuffer.class } ).newInstance( ex );
		} catch ( NoSuchMethodException e ) {
			throw getNonParsableException( "You must provide a constructor " + clazz.getName() + "(StringBuffer).", e );
		} catch ( Exception e ) {
			throw getNonParsableException( "problem at " + clazz.getName() + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(),
					e );
		}
		return erg;
	}
	

	/**
	 * The method returns a {@link NonParsableException} with a given error
	 * message for an {@link Exception}.
	 * 
	 * @param s
	 *            the error message of the {@link NonParsableException}
	 * @param e
	 *            the {@link Exception}
	 * 
	 * @return a {@link NonParsableException} with given error message
	 */
	private static final NonParsableException getNonParsableException( String s, Exception e ) {
		NonParsableException n = new NonParsableException( s );
		n.setStackTrace( e.getStackTrace() );
		return n;
	}

	/**
	 * Returns the value between the tags as {@link String}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a {@link String}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStringAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static String extractStringForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStringAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as {@link String}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a {@link String}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 */
	public static String extractStringAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + tag + "\"." );
		}
		return ex.toString();
	}

	/**
	 * Returns the value between the tags as <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a <code>boolean</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractBooleanAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static boolean[] extractBooleanArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractBooleanArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a <code>boolean</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractBooleanAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static boolean[] extractBooleanArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		boolean[] array = new boolean[l];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractBooleanAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a <code>byte</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractByteAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static byte[] extractByteArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractByteArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a <code>byte</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractByteAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static byte[] extractByteArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		byte[] array = new byte[l];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractByteAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as an <code>int</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractIntArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static int[] extractIntArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractIntArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer}
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as an <code>int</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractIntAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static int[] extractIntArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		int[] array = new int[l];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractIntAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a <code>double</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractDoubleArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static double[] extractDoubleArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractDoubleArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a <code>double</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractDoubleAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static double[] extractDoubleArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		double[] array = new double[l];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractDoubleAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 * @see ArrayHandler#cast(Object...)
	 */
	public static Storable[] extractStorableArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorableArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 * @see ArrayHandler#cast(Object...)
	 */
	public static Storable[] extractStorableArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		return extractStorableArrayAndAttributesForTag( source, tag, attributes, filterAttributes, null );
	}
	
	/**
	 * Returns the value between the tags as {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param clazz
	 * 			  the concrete class of the elements of the array, can be <code>null</code> then the class is inferred from the XML,
	 * 			  otherwise it is tried to create an instance of the class without checking whether it is possible. This might
	 * 			  result in an {@link NonParsableException}
	 * 
	 * @return the value between the tags as a {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[] extractStorableArrayForTag( StringBuffer source, String tag, Class<? extends Storable> clazz ) throws NonParsableException {
		return extractStorableArrayAndAttributesForTag( source, tag, null, null, clazz );
	}
	
	/**
	 * Returns the value between the tags as {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * @param clazz
	 * 			  the concrete class of the elements of the array, can be <code>null</code> then the class is inferred from the XML,
	 * 			  otherwise it is tried to create an instance of the class without checking whether it is possible. This might
	 * 			  result in an {@link NonParsableException}
	 * 
	 * @return the value between the tags as a {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static Storable[] extractStorableArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
				Map<String, String> filterAttributes, Class<? extends Storable> clazz ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		Storable[] array;
		if( clazz == null ){
			array = new Storable[l];
		} else {
			array = (Storable[]) Array.newInstance( clazz, l );
		}
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractStorableAndAttributesForTag( ex, "pos", null, myFilterAttributes, clazz );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a {@link String} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStringArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static String[] extractStringArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStringArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a {@link String} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStringAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static String[] extractStringArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		String[] array = new String[l];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractStringAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as two dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a two dimensional {@link Storable}
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[][] extractStorable2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorable2ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as two dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a two dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[][] extractStorable2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
			return extractStorable2ArrayAndAttributesForTag( source, tag, attributes, filterAttributes, null );
	}

	 /**
	 * Returns the value between the tags as two dimensional {@link Storable}
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param clazz
	 *            the concrete class of the elements of the array, can be
	 *            <code>null</code> then the class is inferred from the XML,
	 *            otherwise it is tried to create an instance of the class
	 *            without checking whether it is possible. This might result in
	 *            an {@link NonParsableException}
	 * 
	 * @return the value between the tags as a two dimensional {@link Storable}
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer,
	 *      String, Map, Map)
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[][] extractStorable2ArrayForTag( StringBuffer source, String tag, Class<? extends Storable> clazz ) throws NonParsableException {
			return extractStorable2ArrayAndAttributesForTag( source, tag, null, null, clazz );
	}
	
	 /**
	 * Returns the value between the tags as two dimensional {@link Storable}
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 *            a {@link Map} which can be used to obtain the attribute of the
	 *            tag
	 * @param filterAttributes
	 *            a {@link Map} which defines a filter for the tags
	 * @param clazz
	 *            the concrete class of the elements of the array, can be
	 *            <code>null</code> then the class is inferred from the XML,
	 *            otherwise it is tried to create an instance of the class
	 *            without checking whether it is possible. This might result in
	 *            an {@link NonParsableException}
	 * 
	 * @return the value between the tags as a two dimensional {@link Storable}
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorableArrayAndAttributesForTag(StringBuffer,
	 *      String, Map, Map)
	 */
	public static Storable[][] extractStorable2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes, Class<? extends Storable> clazz ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		Storable[][] array = new Storable[l][];
		if( clazz == null ){
			array = new Storable[l][];
		}
		else {
			array = (Storable[][]) Array.newInstance( clazz, new int[]{ l, 0 } );
		}
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractStorableArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes, clazz );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as three dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a three dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorable3ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable3ArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[][][] extractStorable3ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorable3ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as three dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a three dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable3ArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[][][] extractStorable3ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		return extractStorable3ArrayAndAttributesForTag( source, tag, attributes, filterAttributes, null );
		
	}
	
	/**
	 * Returns the value between the tags as three dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param clazz
	 *            the concrete class of the elements of the array, can be
	 *            <code>null</code> then the class is inferred from the XML,
	 *            otherwise it is tried to create an instance of the class
	 *            without checking whether it is possible. This might result in
	 *            an {@link NonParsableException}
	 * 
	 * @return the value between the tags as a three dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable3ArrayAndAttributesForTag(StringBuffer, String, Map, Map, Class)
	 */
	public static Storable[][][] extractStorable3ArrayForTag( StringBuffer source, String tag, Class<? extends Storable> clazz ) throws NonParsableException {
		return extractStorable3ArrayAndAttributesForTag( source, tag, null, null, clazz );
	}
	
	/**
	 * Returns the value between the tags as three dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * @param clazz
	 *            the concrete class of the elements of the array, can be
	 *            <code>null</code> then the class is inferred from the XML,
	 *            otherwise it is tried to create an instance of the class
	 *            without checking whether it is possible. This might result in
	 *            an {@link NonParsableException}
	 * 
	 * @return the value between the tags as a three dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStorable2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static Storable[][][] extractStorable3ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes, Class<? extends Storable> clazz ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		Storable[][][] array;
		if( clazz == null ) {
			array = new Storable[l][][];
		} else {
			array = (Storable[][][]) Array.newInstance( clazz, new int[]{ l, 0, 0 } );
		}
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractStorable2ArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes, clazz );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as two dimensional <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>boolean</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractBoolean2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static boolean[][] extractBoolean2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractBoolean2ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as two dimensional <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>boolean</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractBooleanArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static boolean[][] extractBoolean2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		boolean[][] array = new boolean[l][];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractBooleanArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as two dimensional <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a two dimensional <code>byte</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractByte2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static byte[][] extractByte2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractByte2ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as two dimensional <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>byte</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractByteArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static byte[][] extractByte2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		byte[][] array = new byte[l][];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractByteArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as two dimensional <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a two dimensional <code>int</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractInt2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static int[][] extractInt2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractInt2ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as two dimensional <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>int</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractIntArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static int[][] extractInt2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		int[][] array = new int[l][];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractIntArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as two dimensional <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>double</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractDouble2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static double[][] extractDouble2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractDouble2ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as two dimensional <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>double</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractDoubleArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static double[][] extractDouble2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		double[][] array = new double[l][];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractDoubleArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Returns the value between the tags as two dimensional {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * 
	 * @return the value between the tags as a two dimensional {@link String}
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractString2ArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static String[][] extractString2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractString2ArrayAndAttributesForTag( source, tag, null, null );
	}

	/**
	 * Returns the value between the tags as two dimensional {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken
	 * @param attributes
	 * 			  a {@link Map} which can be used to obtain the attribute of the tag
	 * @param filterAttributes
	 * 			  a {@link Map} which defines a filter for the tags
	 * 
	 * @return the value between the tags as a two dimensional
	 *         {@link String} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, Map, Map)
	 * @see XMLParser#extractStringArrayAndAttributesForTag(StringBuffer, String, Map, Map)
	 */
	public static String[][] extractString2ArrayAndAttributesForTag( StringBuffer source, String tag, Map<String, String> attributes,
			Map<String, String> filterAttributes ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, tag, attributes, filterAttributes );
		int l = extractIntForTag( ex, "length" );
		String[][] array = new String[l][];
		Map<String, String> myFilterAttributes = new TreeMap<String, String>();
		for( int i = 0; i < l; i++ ) {
			myFilterAttributes.clear();
			myFilterAttributes.put( "val", ""+i );
			array[i] = extractStringArrayAndAttributesForTag( ex, "pos", null, myFilterAttributes );
		}
		return array;
	}

	/**
	 * Frames the {@link StringBuffer} <code>source</code> with equal tags
	 * &quot;&lt; <code>tag</code>&gt;&quot; and &quot;&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param tag
	 *            the tags by which the {@link StringBuffer} should be framed
	 * 
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void addTags( StringBuffer source, String tag ) {
		addTagsAndAttributes( source, tag, null );
	}

	/**
	 * Frames the {@link StringBuffer} <code>source</code> with &quot;&lt;
	 * <code>tag attributes</code>&gt;&quot; and &quot;&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param tag
	 *            the tags by which the {@link StringBuffer} should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void addTagsAndAttributes( StringBuffer source, String tag, String attributes ) {
		source.insert( 0, "<" + tag + (attributes==null?"":(" " + attributes)) +  ">\n" );
		source.insert( source.length(), "</" + tag + ">\n" );
	}

	/**
	 * Appends a <code>boolean</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>b</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendBooleanWithTagsAndAttributes(StringBuffer, boolean, String, String)
	 */
	public static void appendBooleanWithTags( StringBuffer source, boolean b, String tag ) {
		appendBooleanWithTagsAndAttributes( source, b, tag, null );
	}

	/**
	 * Appends a <code>boolean</code> value with the tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag attributes</code>&gt;<code>b</code>&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendBooleanWithTagsAndAttributes( StringBuffer source, boolean b, String tag, String attributes ) {
		appendStringWithTagsAndAttributes( source, "" + b, tag, attributes );
	}

	/**
	 * Appends a <code>byte</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>b</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByteWithTagsAndAttributes(StringBuffer, byte, String, String)
	 */
	public static void appendByteWithTags( StringBuffer source, byte b, String tag ) {
		appendByteWithTagsAndAttributes( source, b, tag, null );
	}

	/**
	 * Appends a <code>byte</code> value with the tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag attributes</code>&gt;<code>b</code>&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by the tags and appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendByteWithTagsAndAttributes( StringBuffer source, byte b, String tag, String attributes ) {
		appendStringWithTagsAndAttributes( source, "" + b, tag, attributes );
	}

	/**
	 * Appends an {@link Enum} object with equal tags to the
	 * {@link StringBuffer} <code>source</code>.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param e
	 *            the enumeration that should be framed by the tags and appended
	 *            to the source
	 * @param tag
	 *            the tags by which the enumeration should be framed
	 * 
	 * @see XMLParser#appendEnumWithTagsAndAttributes(StringBuffer, Enum, String, String)
	 */
	public static <T extends Enum<T>> void appendEnumWithTags( StringBuffer source, Enum<T> e, String tag ) {
		appendEnumWithTagsAndAttributes( source, e, tag, null );
	}

	/**
	 * Appends an {@link Enum} object with the tags to the
	 * {@link StringBuffer} <code>source</code>.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param e
	 *            the enumeration that should be framed by the tags and
	 *            appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendStringWithTagsAndAttributes(StringBuffer, String, String, String)
	 */
	public static <T extends Enum<T>> void appendEnumWithTagsAndAttributes( StringBuffer source, Enum<T> e, String tag, String attributes ) {
		StringBuffer help = new StringBuffer( 1000 );
		appendStringWithTags( help, e.getClass().getName(), "enumName" );
		appendStringWithTags( help, e.name(), "name" );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an <code>int</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>i</code>&lt;<code>/tag</code>&gt;&quot;
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed 
	 * 
	 * @see XMLParser#appendIntWithTagsAndAttributes(StringBuffer, int, String, String)
	 */
	public static void appendIntWithTags( StringBuffer source, int i, String tag ) {
		appendIntWithTagsAndAttributes( source, i, tag, null );
	}

	/**
	 * Appends an <code>int</code> value with the tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag attributes</code>&gt;<code>i</code>&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by the tags and
	 *            appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendIntWithTagsAndAttributes( StringBuffer source, int i, String tag, String attributes ) {
		appendStringWithTagsAndAttributes( source, "" + i, tag, attributes );
	}

	/**
	 * Appends a <code>long</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>i</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendLongWithTagsAndAttributes(StringBuffer, long, String, String)
	 */
	public static void appendLongWithTags( StringBuffer source, long i, String tag ) {
		appendLongWithTagsAndAttributes( source, i, tag, null );
	}

	/**
	 * Appends a <code>long</code> value with the tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag attributes</code>&gt;<code>i</code>&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param l
	 *            the value that should be framed by the tags and
	 *            appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendLongWithTagsAndAttributes( StringBuffer source, long l, String tag, String attributes ) {
		appendStringWithTagsAndAttributes( source, "" + l, tag, attributes );
	}

	/**
	 * Appends a <code>double</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>d</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendDoubleWithTagsAndAttributes(StringBuffer, double, String, String)
	 */
	public static void appendDoubleWithTags( StringBuffer source, double d, String tag ) {
		appendDoubleWithTagsAndAttributes( source, d, tag, null );
	}

	/**
	 * Appends a <code>double</code> value with the tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag attributes</code>&gt;<code>d</code>&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the value that should be framed by the tags and
	 *            appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendDoubleWithTagsAndAttributes( StringBuffer source, double d, String tag, String attributes ) {
		appendStringWithTagsAndAttributes( source, ""+d, tag, attributes );
	}

	/**
	 * Appends a {@link Storable} object or &quot;null&quot; with equal tags to
	 * the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the object that should be framed by the tags and appended to
	 *            the source, if <code>null</code> then &quot;null&quot; framed
	 *            by the tags is appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendStorableWithTagsAndAttributes(StringBuffer, Storable,
	 *      String, String)
	 */
	public static void appendStorableWithTags( StringBuffer source, Storable s, String tag ) {
		appendStorableWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends a {@link Storable} object or &quot;null&quot; with the
	 * tags to the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the object that should be framed by the tags and
	 *            appended to the source, if <code>null</code> then
	 *            &quot;null&quot; framed by the tags is appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendStorableWithTagsAndAttributes( StringBuffer source, Storable s, String tag, String attributes ) {
		StringBuffer help = new StringBuffer( 1000 );
		if( s != null ) {
			appendStringWithTags( help, s.getClass().getName(), "className" );
			help.append( s.toXML() );
		} else {
			help.append( "null" );
		}
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends a {@link String} with equal tags to the {@link StringBuffer}
	 * <code>source</code> in the following way: &quot;&lt;<code>tag</code>&gt;
	 * <code>s</code>&lt;<code>/tag</code> &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the {@link String} that should be framed by the tags and
	 *            appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendStringWithTagsAndAttributes(StringBuffer, String, String, String)
	 */
	public static void appendStringWithTags( StringBuffer source, String s, String tag ) {
		appendStringWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends a {@link String} with the tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag attributes</code>&gt;<code>s</code>&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the value that should be framed by the tags and
	 *            appended to the source
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 */
	public static void appendStringWithTagsAndAttributes( StringBuffer source, String s, String tag, String attributes ) {
		source.append( "<" + tag + (attributes==null?"":(" " + attributes)) + ">" + s + "</" + tag + ">\n" );
	}

	/**
	 * Encodes a <code>byte</code> array. The encoding is the following way: <br>
	 * &lt;length&gt;b.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;b[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * It is necessary to include such an encoded array in tags or another
	 * array.
	 * 
	 * @param b
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#appendByteWithTagsAndAttributes(StringBuffer, byte, String, String)
	 */
	protected static StringBuffer getByteArrayWithTags( byte[] b ) {
		int l = b.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendByteWithTagsAndAttributes( source, b[j], "pos", "val=\"" + j + "\"" );
		}
		return source;
	}

	/**
	 * Encodes an <code>int</code> array. The encoding is like in
	 * {@link #getByteArrayWithTags(byte[])}.
	 * 
	 * @param i
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#appendIntWithTagsAndAttributes(StringBuffer, int, String, String)
	 */
	protected static StringBuffer getIntArrayWithTags( int[] i ) {
		int l = i.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendIntWithTagsAndAttributes( source, i[j], "pos", "val=\"" + j + "\"" );
		}
		return source;
	}

	/**
	 * Encodes a <code>double</code> array. The encoding is like in
	 * {@link #getByteArrayWithTags(byte[])}.
	 * 
	 * @param d
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#appendDoubleWithTagsAndAttributes(StringBuffer, double, String, String)
	 */
	protected static StringBuffer getDoubleArrayWithTags( double[] d ) {
		int l = d.length;
		StringBuffer source = new StringBuffer( 25 + l * 40 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendDoubleWithTagsAndAttributes( source, d[j], "pos", "val=\"" + j + "\"" );
		}
		return source;
	}

	/**
	 * Encodes a <code>boolean</code> array. The encoding is like in
	 * {@link #getByteArrayWithTags(byte[])}.
	 * 
	 * @param b
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#appendBooleanWithTagsAndAttributes(StringBuffer, boolean, String,
	 *      String)
	 */
	protected static StringBuffer getBooleanArrayWithTags( boolean[] b ) {
		int l = b.length;
		StringBuffer source = new StringBuffer( 25 + l * 40 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendBooleanWithTagsAndAttributes( source, b[j], "pos", "val=\"" + j + "\"" );
		}
		return source;
	}

	/**
	 * Encodes a {@link String} array. The encoding is like in
	 * {@link #getByteArrayWithTags(byte[])}.
	 * 
	 * @param s
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#appendStringWithTagsAndAttributes(StringBuffer, String, String, String)
	 */
	protected static StringBuffer getStringArrayWithTags( String[] s ) {
		int l = s.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendStringWithTagsAndAttributes( source, s[j], "pos", "val=\"" + j + "\"" );
		}
		return source;
	}

	/**
	 * Encodes a {@link Storable} array. The encoding is like in
	 * {@link #getByteArrayWithTags(byte[])}.
	 * 
	 * @param s
	 *            the {@link Storable} object that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded {@link Storable}
	 *         object
	 * 
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#appendStorableWithTagsAndAttributes(StringBuffer, Storable, String,
	 *      String)
	 */
	protected static StringBuffer getStorableArrayWithTags( Storable[] s ) {
		int l = s.length;
		StringBuffer source = new StringBuffer( 25 + l * 1030 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendStorableWithTagsAndAttributes( source, s[j], "pos", "val=\"" + j + "\"" );
		}
		return source;
	}

	/**
	 * Appends an encoded <code>boolean</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 *            
	 * @see XMLParser#appendBooleanArrayWithTagsAndAttributes(StringBuffer, boolean[],
	 *      String, String)
	 */
	public static void appendBooleanArrayWithTags( StringBuffer source, boolean[] b, String tag ) {
		appendBooleanArrayWithTagsAndAttributes( source, b, tag, null );
	}

	/**
	 * Appends an encoded <code>boolean</code> array with the tags to
	 * the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#getBooleanArrayWithTags(boolean[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendBooleanArrayWithTagsAndAttributes( StringBuffer source, boolean[] b, String tag, String attributes ) {
		StringBuffer help = getBooleanArrayWithTags( b );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an encoded <code>byte</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 */
	public static void appendByteArrayWithTags( StringBuffer source, byte[] b, String tag ) {
		appendByteArrayWithTagsAndAttributes( source, b, tag, null );
	}

	/**
	 * Appends an encoded <code>byte</code> array with the tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is this way: <br>
	 * &lt;<code>tag attributes</code>&gt; &lt;length&gt;b.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;b[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;<code>/tag</code>&gt;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendByteArrayWithTagsAndAttributes( StringBuffer source, byte[] b, String tag, String attributes ) {
		StringBuffer help = getByteArrayWithTags( b );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an encoded <code>int</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendIntArrayWithTagsAndAttributes(StringBuffer, int[], String,
	 *      String)
	 */
	public static void appendIntArrayWithTags( StringBuffer source, int[] i, String tag ) {
		appendIntArrayWithTagsAndAttributes( source, i, tag, null );
	}

	/**
	 * Appends an encoded <code>int</code> array with the tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#getIntArrayWithTags(int[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendIntArrayWithTagsAndAttributes( StringBuffer source, int[] i, String tag, String attributes ) {
		StringBuffer help = getIntArrayWithTags( i );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an encoded <code>double</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendDoubleArrayWithTagsAndAttributes(StringBuffer, double[], String,
	 *      String)
	 */
	public static void appendDoubleArrayWithTags( StringBuffer source, double[] d, String tag ) {
		appendDoubleArrayWithTagsAndAttributes( source, d, tag, null );
	}

	/**
	 * Appends an encoded <code>double</code> array with the tags to
	 * the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#getDoubleArrayWithTags(double[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendDoubleArrayWithTagsAndAttributes( StringBuffer source, double[] d, String tag, String attributes ) {
		StringBuffer help = getDoubleArrayWithTags( d );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an encoded {@link Storable} array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the {@link Storable} array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded {@link Storable} array should be
	 *            framed
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendStorableArrayWithTagsAndAttributes(StringBuffer, Storable[],
	 *      String, String)
	 */
	public static void appendStorableArrayWithTags( StringBuffer source, Storable[] s, String tag ) {
		appendStorableArrayWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends an encoded {@link Storable} array with the tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the {@link Storable} array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#getStorableArrayWithTags(Storable[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendStorableArrayWithTagsAndAttributes( StringBuffer source, Storable[] s, String tag, String attributes ) {
		StringBuffer help = getStorableArrayWithTags( s );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an encoded {@link String} array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendStringArrayWithTagsAndAttributes(StringBuffer, String[], String,
	 *      String)
	 */
	public static void appendStringArrayWithTags( StringBuffer source, String[] s, String tag ) {
		appendStringArrayWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends an encoded {@link String} array with the tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByteArrayWithTagsAndAttributes(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#getStringArrayWithTags(String[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendStringArrayWithTagsAndAttributes( StringBuffer source, String[] s, String tag, String attributes ) {
		StringBuffer help = getStringArrayWithTags( s );
		addTagsAndAttributes( help, tag, attributes );
		source.append( help );
	}

	/**
	 * Appends an encoded two dimensional <code>boolean</code> array with equal
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional array should be
	 *            framed
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendBoolean2ArrayWithTagsAndAttributes(StringBuffer, boolean[][],
	 *      String, String)
	 */
	public static void appendBoolean2ArrayWithTags( StringBuffer source, boolean[][] i, String tag ) {
		appendBoolean2ArrayWithTagsAndAttributes( source, i, tag, null );
	}

	/**
	 * Appends an encoded two dimensional <code>boolean</code> array with the
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding
	 * is like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendBooleanWithTags(StringBuffer, boolean, String)
	 * @see XMLParser#getBooleanArrayWithTags(boolean[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendBoolean2ArrayWithTagsAndAttributes( StringBuffer source, boolean[][] i, String tag, String attributes ) {
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = getBooleanArrayWithTags( i[j] );
			help2.append( "\t" );
			addTagsAndAttributes( help2, "pos", "val=\"" + j + "\"" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional <code>byte</code> array with equal
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 */
	public static void appendByte2ArrayWithTags( StringBuffer source, byte[][] b, String tag ) {
		appendByte2ArrayWithTagsAndAttributes( source, b, tag, null );
	}

	/**
	 * Appends an encoded two dimensional <code>byte</code> array with the
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * the following way: <br>
	 * &lt;tag attributes&gt; &lt;length&gt;b.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt; &lt;length&gt;b[0].length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0][0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;bi[0][1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt; &lt;length&gt;b[1].length&lt;/length&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/tag&gt;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#getByteArrayWithTags(byte[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendByte2ArrayWithTagsAndAttributes( StringBuffer source, byte[][] b, String tag, String attributes ) {
		int l = b.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = getByteArrayWithTags( b[j] );
			help2.append( "\t" );
			addTagsAndAttributes( help2, "pos", "val=\"" + j + "\"" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional <code>int</code> array with equal tags
	 * to the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendInt2ArrayWithTagsAndAttributes(StringBuffer, int[][], String,
	 *      String)
	 */
	public static void appendInt2ArrayWithTags( StringBuffer source, int[][] i, String tag ) {
		appendInt2ArrayWithTagsAndAttributes( source, i, tag, null );
	}

	/**
	 * Appends an encoded two dimensional <code>int</code> array with the
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#getIntArrayWithTags(int[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendInt2ArrayWithTagsAndAttributes( StringBuffer source, int[][] i, String tag, String attributes ) {
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = getIntArrayWithTags( i[j] );
			help2.append( "\t" );
			addTagsAndAttributes( help2, "pos", "val=\"" + j + "\"" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional <code>double</code> array with equal
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendDouble2ArrayWithTagsAndAttributes(StringBuffer, double[][],
	 *      String, String)
	 */
	public static void appendDouble2ArrayWithTags( StringBuffer source, double[][] d, String tag ) {
		appendDouble2ArrayWithTagsAndAttributes( source, d, tag, null );
	}

	/**
	 * Appends an encoded two dimensional <code>double</code> array with the
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding
	 * is like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#getDoubleArrayWithTags(double[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendDouble2ArrayWithTagsAndAttributes( StringBuffer source, double[][] d, String tag, String attributes ) {
		int l = d.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = getDoubleArrayWithTags( d[j] );
			help2.append( "\t" );
			addTagsAndAttributes( help2, "pos", "val=\"" + j + "\"" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional {@link Storable} array with equal tags
	 * to the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendStorable2ArrayWithTagsAndAttributes(StringBuffer, Storable[][],
	 *      String, String)
	 */
	public static void appendStorable2ArrayWithTags( StringBuffer source, Storable[][] s, String tag ) {
		appendStorable2ArrayWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends an encoded two dimensional {@link Storable} array with the
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#getStorableArrayWithTags(Storable[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendStorable2ArrayWithTagsAndAttributes( StringBuffer source, Storable[][] s, String tag, String attributes ) {
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = getStorableArrayWithTags( s[j] );
			help2.append( "\t" );
			addTagsAndAttributes( help2, "pos", "val=\"" + j + "\"" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}

	/**
	 * Appends an encoded three dimensional {@link Storable} array with equal
	 * tags to the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the three dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendStorable3ArrayWithTagsAndAttributes(StringBuffer, Storable[][][],
	 *      String, String)
	 */
	public static void appendStorable3ArrayWithTags( StringBuffer source, Storable[][][] s, String tag ) {
		appendStorable3ArrayWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends an encoded three dimensional {@link Storable} array with the
	 * tags to the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the three dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendStorable2ArrayWithTagsAndAttributes(StringBuffer, Storable[][],
	 *      String, String)
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendStorable3ArrayWithTagsAndAttributes( StringBuffer source, Storable[][][] s, String tag, String attributes ) {
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 );
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			appendStorable2ArrayWithTagsAndAttributes( help1, s[j], "pos", "val=\"" + j + "\"" );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional {@link String} array with equal tags
	 * to the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendString2ArrayWithTagsAndAttributes(StringBuffer, String[][],
	 *      String, String)
	 */
	public static void appendString2ArrayWithTags( StringBuffer source, String[][] s, String tag ) {
		appendString2ArrayWithTagsAndAttributes( source, s, tag, null );
	}

	/**
	 * Appends an encoded two dimensional {@link String} array with the
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the value should be framed
	 * @param attributes
	 *            <code>null<code> or some attributes, i.e. <code>value=&quot;100&quot; confidence=&quot;0&quot;</code>
	 * 
	 * @see XMLParser#appendByte2ArrayWithTagsAndAttributes(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#getStringArrayWithTags(String[])
	 * @see XMLParser#addTagsAndAttributes(StringBuffer, String, String)
	 */
	public static void appendString2ArrayWithTagsAndAttributes( StringBuffer source, String[][] s, String tag, String attributes ) {
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = getStringArrayWithTags( s[j] );
			help2.append( "\t" );
			addTagsAndAttributes( help2, "pos", "val=\"" + j + "\"" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTagsAndAttributes( help1, tag, attributes );
		source.append( help1 );
	}
}
