/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This is the main class for all {@link ScoringFunction}s that allow to score
 * subsequences of arbitrary length. This {@link ScoringFunction} should be the
 * super class for non-motif {@link ScoringFunction}s like homogeneous Markov
 * models, cyclic Markov models, ... etc.
 * 
 * @author Jens Keilwagen
 */
public abstract class VariableLengthScoringFunction extends
		AbstractNormalizableScoringFunction {

	/**
	 * This is the main constructor that creates an instance of a
	 * {@link VariableLengthScoringFunction} that models sequences of arbitrary
	 * length.
	 * 
	 * @param alphabets
	 *            the {@link AlphabetContainer} of this
	 *            {@link VariableLengthScoringFunction}
	 * 
	 * @see VariableLengthScoringFunction#VariableLengthScoringFunction(AlphabetContainer,
	 *      int)
	 */
	protected VariableLengthScoringFunction(AlphabetContainer alphabets) {
		this(alphabets, 0);
	}

	/**
	 * This is the main constructor that creates an instance of a
	 * {@link VariableLengthScoringFunction} that models sequences of a given
	 * length.
	 * 
	 * @param alphabets
	 *            the {@link AlphabetContainer} of this
	 *            {@link VariableLengthScoringFunction}
	 * @param length
	 *            the length of the modeled sequences
	 */
	protected VariableLengthScoringFunction(AlphabetContainer alphabets,
			int length) {
		super(alphabets, length);
		if (!alphabets.isSimple() || !alphabets.isDiscrete()) {
			throw new IllegalArgumentException(
					"The AlphabetContainer has to be simple and discrete.");
		}
	}

	/**
	 * This is the constructor for the interface {@link de.jstacs.Storable}.
	 * Creates a new {@link VariableLengthScoringFunction} out of its XML
	 * representation.
	 * 
	 * @param source
	 *            the XML representation as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the XML representation could not be parsed.
	 */
	protected VariableLengthScoringFunction(StringBuffer source)
			throws NonParsableException {
		super(source);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seede.jstacs.scoringFunctions.NormalizableScoringFunction#
	 * getNormalizationConstant()
	 */
	public double getNormalizationConstant() {
		return getNormalizationConstant(length);
	}

	/**
	 * This method returns the normalization constant for a given sequence
	 * length.
	 * 
	 * @param length
	 *            the sequence length
	 * 
	 * @return the normalization constant
	 * 
	 * @see NormalizableScoringFunction#getNormalizationConstant()
	 */
	public abstract double getNormalizationConstant(int length);

	/*
	 * (non-Javadoc)
	 * 
	 * @seede.jstacs.scoringFunctions.NormalizableScoringFunction#
	 * getPartialNormalizationConstant(int)
	 */
	public double getPartialNormalizationConstant(int parameterIndex)
			throws Exception {
		return getPartialNormalizationConstant(parameterIndex, length);
	}

	/**
	 * This method returns the partial normalization constant for a given
	 * parameter index and a sequence length.
	 * 
	 * @param parameterIndex
	 *            the index of the parameter
	 * @param length
	 *            the sequence length
	 * 
	 * @return the partial normalization constant
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see NormalizableScoringFunction#getPartialNormalizationConstant(int)
	 */
	public abstract double getPartialNormalizationConstant(int parameterIndex,
			int length) throws Exception;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.jstacs.scoringFunctions.ScoringFunction#getLogScore(de.jstacs.data
	 * .Sequence, int)
	 */
	public double getLogScore(Sequence seq, int start) {
		if (length != 0) {
			return getLogScore(seq, start, length);
		} else {
			return getLogScore(seq, start, seq.getLength() - start);
		}
	}

	/**
	 * This method computes the logarithm of the score for a given subsequence.
	 * 
	 * @param seq
	 *            the {@link Sequence}
	 * @param start
	 *            the start index in the {@link Sequence}
	 * @param length
	 *            the length of the {@link Sequence} beginning at <code>start</code>
	 * 
	 * @return the logarithm of the score for the subsequence
	 * 
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getLogScore(Sequence,
	 *      int)
	 */
	public abstract double getLogScore(Sequence seq, int start, int length);

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.jstacs.scoringFunctions.ScoringFunction#getLogScoreAndPartialDerivation
	 * (de.jstacs.data.Sequence, int, de.jstacs.utils.IntList,
	 * de.jstacs.utils.DoubleList)
	 */
	public double getLogScoreAndPartialDerivation(Sequence seq, int start,
			IntList indices, DoubleList dList) {
		if (length != 0) {
			return getLogScoreAndPartialDerivation(seq, start, length, indices,
					dList);
		} else {
			return getLogScoreAndPartialDerivation(seq, start, seq.getLength()
					- start, indices, dList);
		}
	}

	/**
	 * This method computes the logarithm of the score and the partial
	 * derivations for a given subsequence.
	 * 
	 * @param seq
	 *            the {@link Sequence}
	 * @param start
	 *            the start index in the {@link Sequence}
	 * @param length
	 *            the end index in the {@link Sequence}
	 * @param indices
	 *            an {@link IntList} of indices, after method invocation the
	 *            list should contain the indices i where
	 *            <code>\frac{\partial \log
	 *            score(seq,start,length)}{\partial \lambda_i}</code> is not zero
	 * @param dList
	 *            a {@link DoubleList} of partial derivations, after method
	 *            invocation the list should contain the corresponding
	 *            <code>\frac{\partial \log
	 *            score(seq,start,length)}{\partial \lambda_i}</code>
	 * 
	 * @return the logarithm of the score
	 * 
	 * @see ScoringFunction#getLogScoreAndPartialDerivation(Sequence, int,
	 *      IntList, DoubleList)
	 */
	public abstract double getLogScoreAndPartialDerivation(Sequence seq,
			int start, int length, IntList indices, DoubleList dList);

	/**
	 * This method returns the stationary conditional distributions. The first
	 * dimension of the result is used for the order, the second is used for
	 * encoding the context, and the third is used for the different values of
	 * the random variable.
	 * 
	 * <br>
	 * <br>
	 * 
	 * For an homogeneous Markov model of order 2 it returns an array containing
	 * the stationary symbol distribution as first entry, the conditional
	 * stationary distribution of order 1 as second entry and the conditional
	 * distribution of order 2 as third entry.
	 * 
	 * <br>
	 * <br>
	 * This method is used to determine the contrast for
	 * {@link de.jstacs.motifDiscovery.Mutable#determineNotSignificantPositions(double, double[], double[],
			double[][][][], double[][][][], double)}
	 * .
	 * 
	 * @return all conditional stationary distributions
	 * 
	 * @see de.jstacs.motifDiscovery.Mutable
	 */
	public abstract double[][][] getAllConditionalStationaryDistributions();

	/**
	 * This method sets the hyperparameters for the model parameters by
	 * evaluating the given statistic. The statistic can be interpreted as
	 * follows: The model has seen a number of sequences. From these sequences
	 * it is only known how long (<code>length</code>) and how often (
	 * <code>weight</code>) they have been seen.
	 * 
	 * @param length
	 *            the non-negative lengths of the sequences
	 * @param weight
	 *            the non-negative weight for the corresponding sequence
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see de.jstacs.motifDiscovery.Mutable
	 */
	public abstract void setStatisticForHyperparameters(int[] length,
			double[] weight) throws Exception;
}