/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data;

import javax.naming.OperationNotSupportedException;

import de.jstacs.WrongAlphabetException;
import de.jstacs.data.alphabets.ComplementableDiscreteAlphabet;
import de.jstacs.data.sequences.ArbitrarySequence;
import de.jstacs.data.sequences.ByteSequence;
import de.jstacs.data.sequences.DiscreteSequence;
import de.jstacs.data.sequences.IntSequence;
import de.jstacs.data.sequences.RecursiveSequence;
import de.jstacs.data.sequences.ShortSequence;
import de.jstacs.data.sequences.WrongSequenceTypeException;
import de.jstacs.data.sequences.annotation.SequenceAnnotation;

/**
 * This is the main class for all sequences. All sequences are immutable.
 * 
 * @author Jens Keilwagen
 */
public abstract class Sequence implements Comparable<Sequence> {

	/**
	 * The underlying alphabets.
	 */
	protected AlphabetContainer alphabetCon;

	/**
	 * The pointer to the reverse complement of the {@link Sequence}.
	 */
	protected Sequence rc;

	/**
	 * The annotation of the {@link Sequence}.
	 */
	protected SequenceAnnotation[] annotation;

	/**
	 * Creates a new {@link Sequence} with the given {@link AlphabetContainer}
	 * and the given annotation, but without the content. The content has to be
	 * set by the constructor of the extending class.
	 * 
	 * @param container
	 *            the {@link AlphabetContainer} of the {@link Sequence}
	 * @param annotation
	 *            the annotation of the {@link Sequence}
	 */
	protected Sequence( AlphabetContainer container, SequenceAnnotation[] annotation ) {
		if( container == null ) {
			throw new NullPointerException();
		}
		this.alphabetCon = container;
		if( annotation != null ) {
			this.annotation = annotation.clone();
		}
	}

	/**
	 * Returns the continuous value at position <code>pos</code> of the
	 * {@link Sequence}.
	 * 
	 * @param pos
	 *            the position of the {@link Sequence}
	 * 
	 * @return the continuous value at position <code>pos</code> of the
	 *         {@link Sequence}
	 */
	public abstract double continuousVal( int pos );

	/**
	 * Returns the discrete value at position <code>pos</code> of the
	 * {@link Sequence}.
	 * 
	 * @param pos
	 *            the position of the {@link Sequence}
	 * 
	 * @return the discrete value at position <code>pos</code> of the
	 *         {@link Sequence}
	 */
	public abstract int discreteVal( int pos );

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals( Object o ) {
		if( o == null ) {
			return false;
		}
		if( !( o instanceof Sequence ) ) {
			return false;
		}
		Sequence s = (Sequence)o;
		return compareTo( s ) == 0 && alphabetCon.checkConsistency( s.alphabetCon );
	}

	/**
	 * Return the alphabets, i.e. the {@link AlphabetContainer}, used in this
	 * {@link Sequence}.
	 * 
	 * @return the alphabets, i.e. the {@link AlphabetContainer}, used in this
	 *         {@link Sequence}
	 */
	public final AlphabetContainer getAlphabetContainer() {
		return alphabetCon;
	}

	/**
	 * Returns the annotation of the {@link Sequence}.
	 * 
	 * @return the annotation of the {@link Sequence} (can be <code>null</code>)
	 */
	public final SequenceAnnotation[] getAnnotation() {
		if( annotation != null ) {
			return annotation.clone();
		} else {
			return null;
		}
	}

	/**
	 * This method should be used if one wants to create a {@link Sample} of
	 * {@link CompositeSequence}s. With this constructor you are enabled to
	 * create a {@link Sample} where every {@link Sequence} has the same
	 * {@link AlphabetContainer} instance.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Internally it is checked that the {@link AlphabetContainer} matches with
	 * the one of the {@link CompositeSequence}.
	 * 
	 * @param abc
	 *            the new {@link AlphabetContainer}
	 * @param starts
	 *            the start positions of the junks
	 * @param lengths
	 *            the length of each junk
	 * 
	 * @return the {@link CompositeSequence}
	 * 
	 * @see Sequence.CompositeSequence#Sequence.CompositeSequence(de.jstacs.data.AlphabetContainer,
	 * 			de.jstacs.data.Sequence, int[], int[]) CompositeSequence#CompositeSequence(de.jstacs.data.AlphabetContainer,
	 * 			de.jstacs.data.Sequence, int[], int[])
	 */
	public Sequence getCompositeSequence( AlphabetContainer abc, int[] starts, int[] lengths ) {
		return new CompositeSequence( abc, this, starts, lengths );
	}

	/**
	 * This is a very efficient way to create a {@link CompositeSequence} for
	 * sequences with a simple {@link AlphabetContainer}.
	 * 
	 * @param starts
	 *            the start positions of the junks
	 * @param lengths
	 *            the length of each junk
	 * 
	 * @return the {@link CompositeSequence}
	 * 
	 * @see Sequence.CompositeSequence#Sequence.CompositeSequence(de.jstacs.data.Sequence, int[], int[]) CompositeSequence#CompositeSequence(de.jstacs.data.Sequence, int[], int[])
	 */
	public Sequence getCompositeSequence( int[] starts, int[] lengths ) {
		return new CompositeSequence( this, starts, lengths );
	}

	/**
	 * This method should be used if one wants to create a {@link Sample} of
	 * subsequences of defined length. With this constructor you are enabled to
	 * create a {@link Sample} where every {@link Sequence} has the same
	 * {@link AlphabetContainer} instance.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Internally it is checked that the {@link AlphabetContainer} matches with
	 * the one of the subsequence.
	 * 
	 * @param abc
	 *            the new {@link AlphabetContainer}
	 * @param start
	 *            the index of the start position
	 * 
	 * @return the subsequence
	 * 
	 * @see Sequence#getSubSequence(de.jstacs.data.AlphabetContainer, int, int)
	 */
	public final Sequence getSubSequence( AlphabetContainer abc, int start ) {
		return getSubSequence( abc, start, getLength() - start );
	}

	/**
	 * This method should be used if one wants to create a {@link Sample} of
	 * subsequences of defined length. With this constructor you are enabled to
	 * create a {@link Sample} where every {@link Sequence} has the same
	 * {@link AlphabetContainer} instance.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Internally it is checked that the {@link AlphabetContainer} matches with
	 * the one of the subsequence.
	 * 
	 * @param abc
	 *            the new {@link AlphabetContainer}
	 * @param start
	 *            the index of the start position
	 * @param length
	 *            the length of the new {@link Sequence}
	 * 
	 * @return the subsequence
	 * 
	 * @see Sequence.SubSequence#Sequence.SubSequence(de.jstacs.data.AlphabetContainer, de.jstacs.data.Sequence, int, int) SubSequence#SubSequence(de.jstacs.data.AlphabetContainer, de.jstacs.data.Sequence, int, int)
	 */
	public Sequence getSubSequence( AlphabetContainer abc, int start, int length ) {
		if( start == 0 && length == getLength() ) {
			return this;
		} else {
			return new SubSequence( abc, this, start, length );
		}
	}

	/**
	 * This is a very efficient way to create a subsequence/suffix for
	 * {@link Sequence}s with a simple {@link AlphabetContainer}.
	 * 
	 * @param start
	 *            the index of the start position
	 * 
	 * @return the subsequence
	 * 
	 * @see Sequence#getSubSequence(int, int)
	 */
	public final Sequence getSubSequence( int start ) {
		return getSubSequence( start, getLength() - start );
	}

	/**
	 * This is a very efficient way to create a subsequence of defined length
	 * for {@link Sequence}s with a simple {@link AlphabetContainer}.
	 * 
	 * @param start
	 *            the index of the start position
	 * @param length
	 *            the length of the new {@link Sequence}
	 * 
	 * @return the subsequence
	 * 
	 * @see Sequence.SubSequence#Sequence.SubSequence(Sequence, int, int) SubSequence#SubSequence(Sequence, int, int)
	 */
	public Sequence getSubSequence( int start, int length ) {
		if( start == 0 && length == getLength() ) {
			return this;
		} else {
			return new SubSequence( this, start, length );
		}
	}

	/**
	 * This method allows to append annotation to a {@link Sequence}.
	 * 
	 * @param add
	 *            indicates whether to add the new annotation to the existing or
	 *            not
	 * @param annotation
	 *            the new annotation
	 * 
	 * @return the new annotated {@link Sequence}
	 * 
	 * @see Sequence#flatCloneWithoutAnnotation()
	 */
	public Sequence annotate( boolean add, SequenceAnnotation... annotation ) {
		Sequence seq = this.flatCloneWithoutAnnotation();
		if( add ) {
			int num = 0;
			if( annotation != null ) {
				num += annotation.length;
			}
			if( this.annotation != null ) {
				num += this.annotation.length;
			}
			SequenceAnnotation[] temp = new SequenceAnnotation[num];
			num = 0;
			for( ; annotation != null && num < annotation.length; num++ ) {
				temp[num] = annotation[num];
			}
			for( int i = 0; this.annotation != null && i < this.annotation.length; i++ ) {
				temp[num + i] = this.annotation[i];
			}
			seq.annotation = temp;
		} else {
			if( annotation != null ) {
				seq.annotation = annotation.clone();
			} else {
				seq.annotation = null;
			}
		}
		return seq;
	}

	/**
	 * Works in analogy to {@link Sequence#clone()}, but does not clone the
	 * annotation. This method is used in
	 * {@link #annotate(boolean, SequenceAnnotation...)}.
	 * 
	 * @return the cloned {@link Sequence} without annotation
	 */
	protected abstract Sequence flatCloneWithoutAnnotation();

	/**
	 * Returns the length of the {@link Sequence}.
	 * 
	 * @return the length of the {@link Sequence}
	 */
	public abstract int getLength();

	/**
	 * Returns a {@link String} representation of the {@link Sequence} (normally
	 * the {@link Sequence} in its original {@link Alphabet}).
	 * 
	 * @return the {@link Sequence} as {@link String}
	 * 
	 * @see Sequence#toString(String, int, int)
	 */
	@Override
	public String toString() {
		return toString( alphabetCon.getDelim(), 0, getLength() );
	}

	/**
	 * Returns a {@link String} representation of the {@link Sequence} (normally
	 * the {@link Sequence} in its original {@link Alphabet}) beginning at
	 * position <code>start</code> with a default delimiter as separator.
	 * 
	 * @param start
	 *            the start index (inclusive)
	 * 
	 * @return the {@link Sequence} as {@link String}
	 * 
	 * @see Sequence#toString(String, int, int)
	 */
	public String toString( int start ) {
		return toString( alphabetCon.getDelim(), start, getLength() );
	}

	/**
	 * Returns a {@link String} representation of the {@link Sequence} (normally
	 * the {@link Sequence} in its original {@link Alphabet}) between
	 * <code>start</code> and <code>end</code> with a default delimiter as
	 * separator.
	 * 
	 * @param start
	 *            the start index (inclusive)
	 * @param end
	 *            the end index (exclusive)
	 * 
	 * @return the {@link Sequence} as {@link String}
	 * 
	 * @see Sequence#toString(String, int, int)
	 */
	public String toString( int start, int end ) {
		return toString( alphabetCon.getDelim(), start, end );
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo( Sequence s ) {
		//old return this.toString().compareTo( s.toString() );
		int c = alphabetCon.compareTo( s.alphabetCon );
		if( c == 0 ) {
			int l = getLength(), seqL = s.getLength();
			if( l == seqL ) {
				int i = 0;
				if( alphabetCon.isDiscrete() ) {
					int thisVal = 0, seqVal = 0;
					while( i < l && ( thisVal = discreteVal( i ) ) == ( seqVal = s.discreteVal( i ) ) ) {
						i++;
					}
					return thisVal - seqVal;
				} else {
					double thisVal = 0, seqVal = 0;
					while( i < l && ( thisVal = continuousVal( i ) ) == ( seqVal = s.continuousVal( i ) ) ) {
						i++;
					}
					return (int)Math.signum( thisVal - seqVal );
				}
			} else {
				return l - seqL;
			}
		} else {
			return c;
		}
	}

	/**
	 * This method converts a continuous value at position <code>pos</code> of
	 * the {@link Sequence} into a discrete one.
	 * 
	 * @param pos
	 *            the position of the {@link Sequence}
	 * @param content
	 *            the value at this position
	 * 
	 * @return the discrete value for this position
	 * 
	 * @see AlphabetContainer#toDiscrete(int, double)
	 */
	protected int toDiscrete( int pos, double content ) {
		return alphabetCon.toDiscrete( pos, content );
	}

	/**
	 * Returns a {@link String} representation of the {@link Sequence} (normally
	 * the {@link Sequence} in its original alphabet) between <code>start</code>
	 * and <code>end</code> with <code>delim</code> as separator.
	 * 
	 * @param delim
	 *            the delimiter/separator
	 * @param start
	 *            the start index (inclusive)
	 * @param end
	 *            the end index (exclusive)
	 * 
	 * @return the {@link Sequence} as {@link String}
	 */
	public String toString( String delim, int start, int end ) {
		int i = start, l = end - start;
		StringBuffer buf = new StringBuffer( l * ( delim.length() == 0 ? 1 : 6 ) );

		if( l > 0 ) {
			buf.append( alphabetCon.getSymbol( i, continuousVal( i++ ) ) );
			while( i < end ) {
				buf.append( delim + alphabetCon.getSymbol( i, continuousVal( i++ ) ) );
			}
		}
		return buf.toString();
	}

	/**
	 * Creates a {@link Sequence} from a {@link String} based on the given
	 * {@link AlphabetContainer} using the standard delimiter for this
	 * {@link AlphabetContainer}.
	 * 
	 * @param con
	 *            the {@link AlphabetContainer}
	 * @param sequence
	 *            the {@link String} containing the {@link Sequence}
	 * 
	 * @return a new {@link Sequence} instance
	 * 
	 * @throws WrongAlphabetException
	 *             if <code>sequence</code> is not defined over <code>con</code>
	 * @throws IllegalArgumentException
	 *             if the delimiter is empty and the {@link AlphabetContainer}
	 *             is not discrete
	 * 
	 * @see Sequence#create(AlphabetContainer, String, String)
	 */
	public static Sequence create( AlphabetContainer con, String sequence ) throws WrongAlphabetException, IllegalArgumentException {
		return create( con, sequence, con.getDelim() );
	}

	/**
	 * Creates a {@link Sequence} from a {@link String} based on the given
	 * {@link AlphabetContainer} using the given delimiter <code>delim</code>.
	 * 
	 * @param con
	 *            the {@link AlphabetContainer}
	 * @param sequence
	 *            the {@link String} containing the {@link Sequence}
	 * @param delim
	 *            the given delimiter
	 * 
	 * @return a new {@link Sequence} instance
	 * 
	 * @throws WrongAlphabetException
	 *             if <code>sequence</code> is not defined over <code>con</code>
	 * @throws IllegalArgumentException
	 *             if the delimiter is empty and the {@link AlphabetContainer}
	 *             is not discrete
	 * 
	 * @see Sequence#create(AlphabetContainer, SequenceAnnotation[], String,
	 *      String)
	 */
	public static Sequence create( AlphabetContainer con, String sequence, String delim ) throws WrongAlphabetException,
			IllegalArgumentException {
		return create( con, null, sequence, delim );
	}

	/**
	 * Creates a {@link Sequence} from a {@link String} based on the given
	 * {@link AlphabetContainer} using the given delimiter <code>delim</code>
	 * and some <code>annotation</code> for the {@link Sequence}.
	 * 
	 * @param con
	 *            the {@link AlphabetContainer}
	 * @param annotation
	 *            the annotation for the {@link Sequence}
	 * @param sequence
	 *            the {@link String} containing the {@link Sequence}
	 * @param delim
	 *            the given delimiter
	 * 
	 * @return a new {@link Sequence} instance
	 * 
	 * @throws WrongAlphabetException
	 *             if <code>sequence</code> is not defined over <code>con</code>
	 * @throws IllegalArgumentException
	 *             if the delimiter is empty and the {@link AlphabetContainer}
	 *             is not discrete
	 */
	public static Sequence create( AlphabetContainer con, SequenceAnnotation[] annotation, String sequence, String delim ) throws WrongAlphabetException,
			IllegalArgumentException {
		try {
			if( con.isDiscrete() ) {
				// create pure discrete sample
				int l = (int)con.getMaximalAlphabetLength();
				if( l <= Byte.MAX_VALUE ) {
					return new ByteSequence( con, annotation, sequence, delim );
				} else if( l <= Short.MAX_VALUE ) {
					return new ShortSequence( con, annotation, sequence, delim );
				} else if( l <= Integer.MAX_VALUE ) {
					return new IntSequence( con, annotation, sequence, delim );
				} else {
					throw new WrongAlphabetException( "Could not encode. Too many symbols." );
				}
			} else {
				// create hybrid or pure continuous sample
				return new ArbitrarySequence( con, annotation, sequence, delim );
			}
		} catch ( WrongSequenceTypeException e ) {
			RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
			doesNotHappen.setStackTrace( e.getStackTrace() );
			throw doesNotHappen;
		}
	}

	/**
	 * This method returns a new instance of {@link Sequence} containing the
	 * reverse current {@link Sequence}.
	 * 
	 * <br>
	 * 
	 * So invoking this method, for instance, on the sequence &quot;TAATA&quot;
	 * returns &quot;ATAAT&quot;.
	 * 
	 * @return the reverse {@link Sequence}
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current {@link Sequence} is based on an
	 *             {@link AlphabetContainer} that is not simple
	 * 
	 * @see Sequence#reverse(int, int)
	 */
	public final Sequence reverse() throws OperationNotSupportedException {
		return reverse( 0, getLength() );
	}

	/**
	 * This method returns a new instance of {@link Sequence} containing a part
	 * of the reverse current {@link Sequence}.
	 * 
	 * @param start
	 *            the start position (inclusive) in the original
	 *            {@link Sequence}
	 * @param end
	 *            the end position (exclusive) in the original {@link Sequence}
	 * 
	 * @return the reverse {@link Sequence} of the part
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current {@link Sequence} is based on an
	 *             {@link AlphabetContainer} that is not simple
	 */
	public Sequence reverse( int start, int end ) throws OperationNotSupportedException {
		if( alphabetCon.isSimple() ) {
			int i = 0, j = end;
			try {
				if( this instanceof DiscreteSequence ) {
					int[] erg = new int[j - start];
					for( j--; j >= start; j--, i++ ) {
						erg[i] = discreteVal( j );
					}
					return new IntSequence( alphabetCon, erg );
				} else {
					double[] erg = new double[j - start];
					for( j--; j >= start; j--, i++ ) {
						erg[i] = continuousVal( j );
					}
					return new ArbitrarySequence( alphabetCon, erg );
				}
			} catch ( Exception e ) {
				// the current sequence is defined on the correct alphabet => so no WrongAlphabetException can occur
				RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
				doesNotHappen.setStackTrace( e.getStackTrace() );
				throw doesNotHappen;
			}
		} else {
			throw new OperationNotSupportedException( "The sequence has to be simple." );
		}
	}

	/**
	 * This method returns a new instance of {@link Sequence} containing the
	 * complementary current {@link Sequence}.
	 * 
	 * <br>
	 * 
	 * So invoking this method, for instance, on the sequence &quot;TAATA&quot;
	 * with an {@link AlphabetContainer} on
	 * {@link de.jstacs.data.alphabets.DNAAlphabet} returns &quot;ATTAT&quot;.
	 * 
	 * @return the complementary {@link Sequence}
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current {@link Sequence} is not based on a
	 *             {@link ComplementableDiscreteAlphabet}
	 * 
	 * @see ComplementableDiscreteAlphabet
	 * @see Sequence#complement(int, int)
	 */
	public Sequence complement() throws OperationNotSupportedException {
		return complement( 0, getLength() );
	}

	/**
	 * This method returns a new instance of {@link Sequence} containing a part
	 * of the complementary current {@link Sequence}.
	 * 
	 * <br>
	 * 
	 * So invoking this method, for instance, on the sequence &quot;TAATA&quot;
	 * with an {@link AlphabetContainer} on
	 * {@link de.jstacs.data.alphabets.DNAAlphabet} returns &quot;ATTAT&quot;.
	 * 
	 * @param start
	 *            the start position (inclusive) in the original
	 *            {@link Sequence}
	 * @param end
	 *            the end position (exclusive) in the original {@link Sequence}
	 * 
	 * @return the complementary {@link Sequence} of the part
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current {@link Sequence} is not based on a
	 *             {@link ComplementableDiscreteAlphabet}
	 * 
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence complement( int start, int end ) throws OperationNotSupportedException {
		if( alphabetCon.isReverseComplementable() ) {
			ComplementableDiscreteAlphabet cda = (ComplementableDiscreteAlphabet)alphabetCon.getAlphabetAt( 0 );
			try {
				if( cda.length() > 127 ) {
					int[] erg = new int[end - start];
					for( int i = 0, j = start; j < end; i++, j++ ) {
						erg[i] = cda.getComplementaryCode( discreteVal( j ) );
					}

					return new IntSequence( alphabetCon, erg );
				} else {
					byte[] erg = new byte[end - start];
					for( int i = 0, j = start; j < end; i++, j++ ) {
						erg[i] = (byte)cda.getComplementaryCode( discreteVal( j ) );
					}

					return new ByteSequence( alphabetCon, erg );
				}
			} catch ( Exception e ) {
				// the current sequence is defined on the correct alphabet => so no WrongAlphabetException can occur
				RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
				doesNotHappen.setStackTrace( e.getStackTrace() );
				throw doesNotHappen;
			}
		} else {
			throw new OperationNotSupportedException( "The alphabet of sequence has to be complementable." );
		}
	}

	/**
	 * This method returns a new instance of {@link Sequence} containing the
	 * reverse complementary current {@link Sequence}. For more details see the
	 * methods {@link #reverse()} and {@link #complement()}.
	 * 
	 * @return the reverse complementary {@link Sequence}
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current {@link Sequence} is not discrete and simple
	 *             (not based on a {@link ComplementableDiscreteAlphabet})
	 * 
	 * @see Sequence#reverse()
	 * @see Sequence#complement()
	 * @see Sequence#reverseComplement(int, int)
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence reverseComplement() throws OperationNotSupportedException {
		return reverseComplement( 0, getLength() );
	}

	/**
	 * This method returns a new instance of {@link Sequence} containing a
	 * reverse part of the complementary current {@link Sequence}. For more
	 * details see the methods {@link #reverse()} and {@link #complement()}.
	 * 
	 * @param start
	 *            the start position (inclusive) in the original
	 *            {@link Sequence}
	 * @param end
	 *            the end position (exclusive) in the original {@link Sequence}
	 * 
	 * @return the reverse complementary {@link Sequence} of the part
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current {@link Sequence} is not discrete and simple
	 *             ((not based on a {@link ComplementableDiscreteAlphabet})
	 * 
	 * @see Sequence#reverse()
	 * @see Sequence#complement()
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence reverseComplement( int start, int end ) throws OperationNotSupportedException {
		if( rc != null && start == 0 && end == getLength() ) {
			return rc;
		} else if( alphabetCon.isReverseComplementable() ) {
			ComplementableDiscreteAlphabet cda = (ComplementableDiscreteAlphabet)alphabetCon.getAlphabetAt( 0 );
			int i = 0, j = end;
			try {
				Sequence revComp;
				if( cda.length() > 127 ) {
					int[] erg = new int[end - start];
					for( j--; j >= start; j--, i++ ) {
						erg[i] = cda.getComplementaryCode( discreteVal( j ) );
					}

					revComp = new IntSequence( alphabetCon, erg );

				} else {
					byte[] erg = new byte[end - start];
					for( j--; j >= start; j--, i++ ) {
						erg[i] = (byte)cda.getComplementaryCode( discreteVal( j ) );
					}

					revComp = new ByteSequence( alphabetCon, erg );
				}
				if( start == 0 && end == getLength() ) {
					rc = revComp;
					rc.rc = this;
				}
				return revComp;
			} catch ( Exception e ) {
				// the current sequence is defined on the correct alphabet => so no WrongAlphabetException can occur
				RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
				doesNotHappen.setStackTrace( e.getStackTrace() );
				throw doesNotHappen;
			}
		} else {
			throw new OperationNotSupportedException( "The alphabet of sequence has to be reverse-complementable." );
		}
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		int len = getLength();
		if( alphabetCon.isDiscrete() ) {
			//@see String#hashCode()
			int h = 0;
			for( int i = 0; i < len; i++ ) {
				h = 31 * h + discreteVal( i );
			}
			return h;
		} else {
			//@see String#hashCode()
			//@see Double#hashCode()
			double h = 0;
			for( int i = 0; i < len; i++ ) {
				h = 31 * h + continuousVal( i );
			}
			long bits = Double.doubleToLongBits( h );
			return (int)( bits ^ ( bits >>> 32 ) );
		}
	}

	/**
	 * The class handles composite {@link Sequence}s. A
	 * {@link CompositeSequence} consists of several (partial) {@link Sequence}
	 * s. A biological example are promoters like in eukaryotes (-10 and -35
	 * box).
	 * 
	 * @author Jens Keilwagen
	 */
	protected static class CompositeSequence extends RecursiveSequence {

		private static final long serialVersionUID = 1L;

		private int[] starts, lengths;

		private CompositeSequence( AlphabetContainer abc, SequenceAnnotation[] annotation, Sequence seq, int[] starts, int[] lengths,
									boolean check ) {
			super( abc, annotation, seq );
			// TODO make this faster, no new object
			if( check && !abc.checkConsistency( seq.getAlphabetContainer().getCompositeContainer( starts, lengths ) ) ) {
				throw new IllegalArgumentException( "Wrong AlphabetContainer." );
			}
			if( starts.length != lengths.length ) {
				throw new IllegalArgumentException( "starts and lengths have to be from the same dimension" );
			}
			this.starts = starts.clone();
			this.lengths = lengths.clone();
			for( int i = 0; i < lengths.length; i++ ) {
				if( starts[i] < 0 || seq.getLength() <= starts[i] ) {
					throw new IllegalArgumentException( i + "-th start has to be from range [0," + ( seq.getLength() - 1 ) + "]" );
				}
				if( lengths[i] < 0 || seq.getLength() < starts[i] + lengths[i] ) {
					throw new IllegalArgumentException( i + "-th length has to be from range [0," + ( seq.getLength() - starts[i] ) + "]" );
				}
				this.starts[i] = starts[i];
				this.lengths[i] = lengths[i];
			}
		}

		/**
		 * This is a very efficient way to create a {@link CompositeSequence}
		 * for {@link Sequence}s with a simple {@link AlphabetContainer}.
		 * 
		 * @param seq
		 *            the original {@link Sequence}
		 * @param starts
		 *            the start positions of the junks
		 * @param lengths
		 *            the length of each junk
		 */
		public CompositeSequence( Sequence seq, int[] starts, int[] lengths ) {
			this( seq.getAlphabetContainer().getCompositeContainer( starts, lengths ), null, seq, starts, lengths, false );
		}

		/**
		 * This constructor should be used if one wants to create a
		 * {@link Sample} of {@link CompositeSequence}s. With this constructor
		 * you are enabled to create a {@link Sample} where every
		 * {@link Sequence} has the same {@link AlphabetContainer} instance.
		 * 
		 * <br>
		 * <br>
		 * 
		 * Internally it is checked that the {@link AlphabetContainer} matches
		 * with the one of the subsequence.
		 * 
		 * @param abc
		 *            the new {@link AlphabetContainer}
		 * @param seq
		 *            the original {@link Sequence}
		 * @param starts
		 *            the start positions of the junks
		 * @param lengths
		 *            the length of each junk
		 */
		public CompositeSequence( AlphabetContainer abc, Sequence seq, int[] starts, int[] lengths ) {
			this( abc, null, seq, starts, lengths, true );
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.sequences.RecursiveSequence#getIndex(int)
		 */
		@Override
		protected int getIndex( int pos ) {
			int sum = 0, i = 0;
			while( i < lengths.length && sum + lengths[i] <= pos ) {
				sum += lengths[i++];
			}
			return starts[i] + ( pos - sum );
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.Sequence#getLength()
		 */
		@Override
		public int getLength() {
			int sum = 0;
			for( int i = 0; i < lengths.length; i++ ) {
				sum += lengths[i];
			}
			return sum;
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.Sequence#flatCloneWithoutAnnotation()
		 */
		@Override
		protected Sequence flatCloneWithoutAnnotation() {
			return new CompositeSequence( alphabetCon, null, content, starts, lengths, false );
		}
	}

	/**
	 * This class handles subsequences. {@link SubSequence}s are often used to
	 * extract the {@link Sequence} of a sliding window on a long
	 * {@link Sequence}. The class is implemented in such a way that it avoids
	 * chains of {@link SubSequence}s.
	 * 
	 * @author Jens Keilwagen
	 */
	protected static class SubSequence extends RecursiveSequence {

		private static final long serialVersionUID = 1L;

		private int start, length;

		private SubSequence( AlphabetContainer abc, SequenceAnnotation[] annotation, Sequence seq, int start, int length, boolean check ) {
			super( abc, annotation, ( seq instanceof SubSequence ) ? ( (SubSequence)seq ).content : seq );
			// TODO make this faster, no new object
			if( check && !abc.checkConsistency( seq.getAlphabetContainer().getSubContainer( start, length ) ) ) {
				throw new IllegalArgumentException( "Wrong AlphabetContainer." );
			}
			int sl = seq.getLength();
			if( start < 0 || start > sl ) {
				throw new IllegalArgumentException( "Illegal start position: start=" + start + " not in [0," + sl + "]" );
			}
			if( length < 0 || start + length > sl ) {
				throw new IllegalArgumentException( "Illegal length: length=" + length + " not in [0," + ( sl - start ) + "]" );
			}
			if( seq instanceof SubSequence ) {
				SubSequence s = (SubSequence)seq;
				this.start = s.start + start;
			} else {
				this.start = start;
			}
			this.length = length;
		}

		/**
		 * This constructor should be used if one wants to create a
		 * {@link Sample} of {@link SubSequence}s of defined length. With this
		 * constructor you are enabled to create a {@link Sample} where every
		 * {@link Sequence} has the same {@link AlphabetContainer} instance.
		 * 
		 * <br>
		 * <br>
		 * 
		 * Internally it is checked that the {@link AlphabetContainer} matches
		 * with the one of the {@link SubSequence}.
		 * 
		 * @param abc
		 *            the new {@link AlphabetContainer}
		 * @param seq
		 *            the original {@link Sequence}
		 * @param start
		 *            the index of the start position
		 * @param length
		 *            the length of the new sequence
		 */
		public SubSequence( AlphabetContainer abc, Sequence seq, int start, int length ) {
			this( abc, null, ( seq instanceof SubSequence ) ? ( (SubSequence)seq ).content : seq, start, length, true );
		}

		/**
		 * This is a very efficient way to create a {@link SubSequence} of
		 * defined length for {@link Sequence}s with a simple
		 * {@link AlphabetContainer}.
		 * 
		 * @param seq
		 *            the original {@link Sequence}
		 * @param start
		 *            the index of the start position
		 * @param length
		 *            the length of the new {@link Sequence}
		 */
		public SubSequence( Sequence seq, int start, int length ) {
			this( seq.getAlphabetContainer().getSubContainer( start, length ), null, seq, start, length, false );
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.Sequence#reverseComplement(int, int)
		 */
		@Override
		public Sequence reverseComplement( int start, int end ) throws OperationNotSupportedException {
			if( rc != null && start == 0 && end == getLength() ) {
				return rc;
			} else if( content.rc != null ) {
				Sequence revComp = new SubSequence( content.rc, content.rc.getLength() - this.start - end, end - start );
				if( start == 0 && end == getLength() ) {
					rc = revComp;
					rc.rc = this;
				}
				return revComp;
			}
			return super.reverseComplement( start, end );
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.sequences.RecursiveSequence#getIndex(int)
		 */
		@Override
		protected final int getIndex( int pos ) {
			if( pos < 0 || pos >= length ) {
				throw new ArrayIndexOutOfBoundsException( pos );
			}
			return start + pos;
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.sequences.RecursiveSequence#discreteVal(int)
		 */
		@Override
		public final int discreteVal( int pos ) {
			if( pos < 0 || pos >= length ) {
				throw new ArrayIndexOutOfBoundsException( pos );
			}
			return content.discreteVal( start + pos );
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.Sequence#getLength()
		 */
		@Override
		public int getLength() {
			return length;
		}

		/* (non-Javadoc)
		 * @see de.jstacs.data.Sequence#flatCloneWithoutAnnotation()
		 */
		@Override
		protected Sequence flatCloneWithoutAnnotation() {
			return new SubSequence( alphabetCon, null, content, start, length, false );
		}
	}
}