/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions.directedGraphicalModels;

import java.util.LinkedList;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.algorithms.graphs.TopSort;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;
import de.jstacs.io.ArrayHandler;
import de.jstacs.io.XMLParser;
import de.jstacs.utils.Normalisation;
import de.jstacs.utils.random.DirichletMRG;
import de.jstacs.utils.random.DirichletMRGParams;
import de.jtem.numericalMethods.calculus.specialFunctions.Gamma;

/**
 * Class for the tree that represents the context of a {@link Parameter} in a {@link BayesianNetworkScoringFunction}.
 * @author Jan Grau
 *
 */
public class ParameterTree implements Cloneable {

	private int pos;
	private int[] contextPoss;
	private TreeElement root;
	private AlphabetContainer alphabet;
	private int firstParent;
	private int[] firstChildren;
	
	/**
	 * Creates a new {@link ParameterTree} for the parameters at position <code>pos</code> using the parent positions in <code>contextPoss</code>. 
	 * These are used the extract the correct alphabet out of <code>alphabet</code> for every context position. The first parent is the first parent of the random variable
	 * at <code>pos</code> as given by the topological ordering of the network structure of the enclosing {@link BayesianNetworkScoringFunction}. The first children
	 * are the children the random variable at <code>pos</code> is the first parent for.
	 * @param pos the position of the random variable of the parameters in the tree
	 * @param contextPoss the positions of the context
	 * @param alphabet the alphabet of the enclosing {@link BayesianNetworkScoringFunction}
	 * @param firstParent the first parent of this random variable, or <code>-1</code> if the random variable has no parent
	 * @param firstChildren the first children of this random variable
	 */
	public ParameterTree(int pos, int[] contextPoss,AlphabetContainer alphabet, int firstParent, int[] firstChildren){
		this.pos = pos;
		this.contextPoss = contextPoss;
		this.alphabet = alphabet;
		this.firstParent = firstParent;
		this.firstChildren = firstChildren;
		this.root = new TreeElement(0,alphabet);
	}
	
	/**
	 * Re-creates a {@link ParameterTree} from its XML-representation as returned by {@link ParameterTree#toXML()}. {@link ParameterTree} does not
	 * implement the {@link Storable} interface to recycle the {@link AlphabetContainer} of the enclosing {@link BayesianNetworkScoringFunction}, but besides the
	 * different constructor works like any implementation of {@link Storable}.
	 * @param source the XML-representation
	 * @param alphabet the alphabet
	 * @throws NonParsableException is thrown if the XML-code could not be parsed
	 */
	public ParameterTree(StringBuffer source, AlphabetContainer alphabet) throws NonParsableException{
		try{
			source = XMLParser.extractForTag(source, "parameterTree");
			pos = XMLParser.extractIntForTag(source, "pos");
			contextPoss = XMLParser.extractIntArrayForTag(source, "contextPoss");
			root = new TreeElement(XMLParser.extractForTag(source, "root"));
			this.alphabet = alphabet;
			this.firstParent = XMLParser.extractIntForTag(source, "firstParent");
			this.firstChildren = XMLParser.extractIntArrayForTag(source, "firstChildren");
		}catch(NonParsableException e){
			e.printStackTrace();
			throw e;
		}
		
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	public ParameterTree clone() throws CloneNotSupportedException{
		ParameterTree clone = (ParameterTree) super.clone();
		clone.contextPoss = contextPoss.clone();
		clone.root = root.clone();
		clone.firstChildren = firstChildren.clone();
		return clone;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString(){
		StringBuffer all = new StringBuffer();
		all.append("Probabilities at position "+pos+":\n");
		root.appendToBuffer(all, "");
		return all.toString();
	}
	
	/**
	 * Computes the probabilities for a PWM, i.e. the parameters in the tree have an empty context, 
	 * and inserts them into <code>probs</code>. Used by {@link BayesianNetworkScoringFunction#getPWM()}.
	 * @param probs the array to store the probabilities
	 * @throws Exception is thrown if the tree structure does have a non-empty context
	 */
	public void insertProbs(double[] probs) throws Exception{
		root.insertProbs( probs );
	}
	
	/**
	 * Extracts the {@link Parameter}s from the leaves of this tree in left-to-right order (as specified by the order of the alphabet) and returns them as a {@link LinkedList}.
	 * @return the {@link Parameter}s in linear order
	 */
	public LinkedList<Parameter> linearizeParameters(){
		return root.linearizeParameters(new LinkedList<Parameter>());
	}
	
	/**
	 * Returns true if the random variable of this {@link ParameterTree} is a leaf, i.e. it has no children in the network structure of the
	 * enclosing {@link BayesianNetworkScoringFunction}.
	 * @return if this tree is a leaf
	 */
	public boolean isLeaf() {
		return firstChildren.length == 0;
	}
	
	
	/**
	 * Returns the number of parents for the random variable of this {@link ParameterTree} in the network structure of the enclosing {@link BayesianNetworkScoringFunction}.
	 * This corresponds to the length of the context or the depth of the tree.
	 * @return the number of parents.
	 */
	public int getNumberOfParents(){
		return contextPoss.length;
	}
	
	/**
	 * Prints the structure of this tree.
	 */
	public void print(){
		System.out.println("tree "+pos+": ");
		root.print();
	}
	
	/**
	 * Returns the {@link Parameter} that is responsible for the suffix of sequence <code>seq</code> starting at position <code>start</code>.
	 * @param seq the sequence
	 * @param start the first position in the suffix
	 * @return the parameter
	 */
	public Parameter getParameterFor(Sequence seq, int start){
		return root.getParameterFor(seq,start);
	}
	
	/**
	 * Sets the instance of the parameter for symbol <code>symbol</code> and context <code>context</code> to parameter <code>par</code>.
	 * @param symbol the symbol
	 * @param context the context
	 * @param par the new {@link Parameter} instance
	 */
	public void setParameterFor(int symbol, int[][] context, Parameter par){
		root.setParameterFor(0,symbol,context,par);
	}
	
	
	/**
	 * Resets all pre-computed normalization constants.
	 */
	public void invalidateNormalizers(){
		root.invalidateNormalizers();
	}
	
	/**
	 * Computes the forward-part of the normalization constant starting from this {@link ParameterTree}. This is only possible for roots, i.e.
	 * {@link ParameterTree}s that do not have parents in the network structure of the enclosing {@link BayesianNetworkScoringFunction}.
	 * @param trees the array of all trees as from the enclosing {@link BayesianNetworkScoringFunction}
	 * @return the forward-part of the normalization constant
	 * @throws RuntimeException is thrown if this {@link ParameterTree} is not a root
	 */
	public double forward(ParameterTree[] trees) throws RuntimeException{
		if(this.getNumberOfParents() > 0){
			throw new RuntimeException("Forward can only be started at roots.");
		}else{
			//System.out.println();
			return this.getZ(new int[0][2], trees);
		}
	}
	
	
	private double getZ(int[][] context, ParameterTree[] trees) throws RuntimeException {
		return root.getZ(context, new int[this.getNumberOfParents() + 1][2] , trees, 0);
	}
	
	private double getT(int[][] context, ParameterTree[] trees, int[][] order) throws RuntimeException {
		return root.getT(context, firstParent > -1 ? new int[trees[firstParent].contextPoss.length+1][2] : new int[0][2], trees, order, 0);
	}
	
	/**
	 * Starts the computation of the backward-part of the normalization constant starting from this {@link ParameterTree}. This is only possible for leaves, i.e.
	 * {@link ParameterTree}s that do not have children in the network structure of the enclosing {@link BayesianNetworkScoringFunction}.
	 * @param trees the array of all trees as from the enclosing {@link BayesianNetworkScoringFunction}
	 * @param order the topological ordering as returned by {@link TopSort#getTopologicalOrder(int[][])}
	 * @throws RuntimeException is thrown if this {@link ParameterTree} is not a leaf
	 */
	public void backward(ParameterTree[] trees, int[][] order) throws RuntimeException{
		if(! this.isLeaf()){
			throw new RuntimeException("Backward can only be started at leaves.");
		}else{
			root.startBackward(new int[this.getNumberOfParents() + 1][2],trees,order,0);
		}
	}
	
	
	/**
	 * Adds <code>count</code> to the parameter as returned by {@link ParameterTree#getParameterFor(Sequence, int)}.
	 * @param seq the sequence
	 * @param start the first position of the suffix of <code>seq</code
	 * @param count the added count
	 */
	public void addCount(Sequence seq, int start, double count){
		this.getParameterFor(seq, start).addCount(count);
	}
	
	
	/**
	 * Starts the normalization of the plug in parameters to the log of the MAP-estimates.
	 */
	public void normalizePlugInParameters(){
		root.normalizePlugInParameters();
	}
	
	/**
	 * Divides each of the normalized parameters on a simplex by the last parameter, which is defined not to be free.
	 * In the log-space this amounts to subtracting the value of the last parameter from all parameters on the simplex.
	 */
	public void divideByUnfree(){
		root.divideByUnfree();
	}
	
	/**
	 * Works as defined in {@link Storable}.
	 * @return the XML-representation
	 */
	public StringBuffer toXML() {
		StringBuffer source = new StringBuffer();
		XMLParser.appendIntWithTags(source, pos, "pos");
		XMLParser.appendIntArrayWithTags(source, contextPoss, "contextPoss");
		XMLParser.appendStorableWithTags(source, root, "root");
		XMLParser.appendIntWithTags(source, firstParent, "firstParent");
		XMLParser.appendIntArrayWithTags(source, firstChildren, "firstChildren");
		XMLParser.addTags(source, "parameterTree");
		return source;
		
	}
	
	/**
	 * Creates a {@link String} array from the XML-representation of all {@link ParameterTree}s in <code>trees</code> as returned by the
	 * corresponding {@link ParameterTree#toXML()} methods.
	 * @param trees the trees
	 * @return the XML-representations as {@link String} array
	 */
	public static String[] toStringArray(ParameterTree[] trees){
		String[] strs = new String[trees.length];
		for(int i=0;i<trees.length;i++){
			strs[i] = trees[i].toXML().toString();
		}
		return strs;
	}
	
	/**
	 * Re-creates an array of {@link ParameterTree}s from their XML-representation as given in <code>strs</code>.
	 * @param strs the XML-representations as returned by {@link ParameterTree#toStringArray(ParameterTree[])}
	 * @param alphabet the alphabet of the enclosing {@link BayesianNetworkScoringFunction}
	 * @return the {@link ParameterTree}s
	 * @throws NonParsableException is thrown if one of the XML-representations could not be parsed
	 */
	public static ParameterTree[] fromStringArray(String[] strs, AlphabetContainer alphabet) throws NonParsableException{
		ParameterTree[] trees = new ParameterTree[strs.length];
		for(int i=0;i<trees.length;i++){
			trees[i] = new ParameterTree(new StringBuffer(strs[i]),alphabet);
		}
		return trees;
	}
	
	/**
	 * Returns the first parent of the random variable of this {@link ParameterTree} in the topological ordering
	 * of the network structure of the enclosing {@link BayesianNetworkScoringFunction}.
	 * @return the first parent
	 */
	public int getFirstParent() {
		return firstParent;
	}

	/**
	 * Returns the KL-divergence of the parameters of a PWM to the reference distribution <code>q</code>.
	 * @param q the reference
	 * @return the KL-divergence
	 */
	public double getKLDivergence(double[] q) {
		return root.getKLDivergence(q);
	}

	/**
	 * Fills all parameters with the probabilities given in <code>fillEmptyWith</code>.
	 * @param fillEmptyWith the replacement
	 */
	public void fill(double[] fillEmptyWith) {
		root.fill(fillEmptyWith);
	}

	/**
	 * Copies the values of the parameters from another {@link ParameterTree}
	 * @param parameterTree the template
	 */
	public void copy(ParameterTree parameterTree) {
		root.copy(parameterTree.root);
	}

	/**
	 * Initializes the parameters of this {@link ParameterTree} randomly
	 * @param ess the equivalent sample size
	 */
	public void initializeRandomly(double ess) {
		root.initializeRandomly(ess);
	}

	/**
	 * Computes the Gamma-normalization for the prior
	 * @return the Gamma-normalization
	 */
	public Double computeGammaNorm() {
		return root.computeGammaNorm();
	}

	
	private class TreeElement implements Storable, Cloneable{
		
		private int contextPos;
		private TreeElement[] children;
		private Parameter[] pars;
		private Double fullNormalizer;
		private Double[] symT;
		
		private TreeElement(int contNum, AlphabetContainer alphabet){
			if(contNum < contextPoss.length){
				this.contextPos = contextPoss[contNum];
				children = new TreeElement[(int) alphabet.getAlphabetLengthAt(this.contextPos)];
				for(int i=0;i<alphabet.getAlphabetLengthAt(this.contextPos);i++){
					children[i] = new TreeElement(contNum+1,alphabet);
				}
			}else{
				this.contextPos = -1;
				this.pars = new Parameter[(int) alphabet.getAlphabetLengthAt(pos)];

				this.fullNormalizer = null;
				this.symT = new Double[pars.length];
			}
		}
		
		private void appendToBuffer(StringBuffer all, String after){
			if(children != null){
				for(int i=0;i<children.length;i++){
					children[i].appendToBuffer(all,after+"X_"+contextPos+" = "+alphabet.getSymbol(contextPos, i)+", ");
				}
			}else{
				double norm = 0;
				for(int i=0;i<pars.length;i++){
					norm += pars[i].getExpValue()*pars[i].getZ();
				}
				for(int i=0;i<pars.length;i++){
					double tempTheta = pars[i].getExpValue()*pars[i].getZ()/norm;
					all.append("P(X_"+pos+" = "+alphabet.getSymbol(pos, i)+" | "+after+"c)="+tempTheta);
					if(i<pars.length - 1){
						all.append("\t");
					}
				}
				all.append("\n");
			}
		}
		
		private void insertProbs(double[] probs) throws Exception{
			if(children != null){
				throw new Exception("Not implemented");
			}else{
				for(int i=0;i<pars.length;i++){
					probs[i] = pars[i].getValue() + Math.log( pars[i].getZ() );
				}
				Normalisation.logSumNormalisation( probs );
				
			}
		}
		
		private void startBackward(int[][] newContext, ParameterTree[] trees, int[][] order, int depth) throws RuntimeException {
			if(children != null){
				newContext[depth][0] = contextPos;
				for(int i=0;i<children.length;i++){
					newContext[depth][1] = i;
					children[i].startBackward(newContext, trees, order, depth+1);
				}
			}else{
				newContext[depth][0] = pos;
				for(int i=0;i<pars.length;i++){
					newContext[depth][1] = pars[i].symbol;
					trees[pos].getT(newContext, trees, order);
				}
			}
		}

		private double getT(int[][] context, int[][] newContext, ParameterTree[] trees,int[][] order, int depth) throws RuntimeException{
			//we are not at a leaf of the ParameterTree
			if(children != null){
				//search for my context's position
				for(int i=0;i<context.length;i++){
					if(context[i][0] == this.contextPos){
						//copy to newContext, we might need it for the predecessors, if the have the same
						newContext[depth][0] = context[i][0];
						newContext[depth][1] = context[i][1];
						//go to child for found context
						return children[ context[i][1] ].getT(context, newContext, trees,order, depth+1);
					}
				}
				//should not happen
				throw new RuntimeException("Correct context not found for depth "+depth+" at position "+pos+".");
			}else{
				//search for the context that corresponds to my position
				for(int i=0;i<context.length;i++){
					if(context[i][0] == pos){
						//search for parameter that corresponds to realization of found context
						for(int j=0;j<pars.length;j++){
							if(pars[j].symbol == context[i][1]){
								//if already computed for this parameter, done
								if(symT[j] != null){
									return symT[j];
								}else{
									int fp = firstParent;
									//if we are at a root
									if(fp == -1){
										pars[j].setT(1d);
										return pars[j].getExpValue();
									//if we are at a node with a context that fully defines its first parent,
									//i.e. the parent that fully defines the configuration of this node
									}else if(trees[fp].contextPoss.length < contextPoss.length){
										//recursion
										double temp = trees[fp].getT(newContext, trees, order);
										int[] fcoffp = trees[fp].firstChildren;
										//for the other branches under fp that are independent of this node's
										//branch, we need the Zs, which have already been computed
										for(int k=0;k<fcoffp.length;k++){
											//but not for this branch
											if(fcoffp[k] != pos){
												//System.out.println(fp+" has fc "+fcoffp[k]);
												temp *= trees[fcoffp[k]].getZ(newContext, trees);
											}
										}
										//set the T for the partial normalization
										pars[j].setT(temp);
										//compute the T for recursion
										symT[j] = pars[j].getExpValue()*temp;
										return symT[j];
									//we are at a node with a parent that has more and different parents than this node
									//(at most one, this is the parent over whose configs we will sum)
									}else{
										
										//find the parent of the first parent
										//that has the lowest number in the topological order,
										//this must be the parent on which we do not depend
										int[] cp = trees[fp].contextPoss;
										int lowestOrder = Integer.MAX_VALUE;
										int lowestOrderIndex = -1;
										for(int k=0;k<cp.length;k++){
											if(order[cp[k]][1] < lowestOrder){
												lowestOrder = order[cp[k]][1];
												lowestOrderIndex = cp[k];
											}
										}
										
										//sum over configurations of the parent lowestOrderIndex
										newContext[depth][0] = lowestOrderIndex;
										double temp = 0;
										for(byte a = 0; a<alphabet.getAlphabetLengthAt(lowestOrderIndex); a++){
											newContext[depth][1] = a;
											//recursion
											double temp2 = trees[fp].getT(newContext,trees,order);
											//for the other branches under fp that are independent of this node's
											//branch, we need the Zs, which have already been computed
											int[] fcoffp = trees[fp].firstChildren;
											for(int k=0;k<fcoffp.length;k++){
												//but nor for this branch
												if(fcoffp[k] != pos){
													temp2 *= trees[fcoffp[k]].getZ(newContext, trees);
												}
											}
											temp += temp2;
										}
										//set T for partial normalization
										pars[j].setT(temp);
										//compute the T for recursion
										symT[j] = pars[j].getExpValue()*temp;
										return symT[j];
									}
								}
							}
						}
					}
				}
				throw new RuntimeException("Parameter value not defined in context.");
			}
		}

		private double getZ(int[][] context, int[][] newContext, ParameterTree[] trees, int depth) throws RuntimeException{
			//we are not at a leaf of the ParameterTree
			if(children != null){
				//search for my context's position
				for(int i=0;i<context.length;i++){
					//found
					if(context[i][0] == this.contextPos){
						//copy to newContext, we need it for the descendants
						newContext[depth][0] = context[i][0];
						newContext[depth][1] = context[i][1];
						//go to children for found contet
						return children[ context[i][1] ].getZ(context, newContext, trees, depth+1);
					}
				}
				//should not happen
				throw new RuntimeException("Correct context could not be found at position "+pos+" and depth "+depth);
			}else if(fullNormalizer != null){
				//we already precomputed the normalization constant, done
				return fullNormalizer;
			}else{
				double val = 0;
				//to compute the normalization constant we must sum over all children
				for(int i=0;i<pars.length;i++){
					int[] fc = firstChildren;
					if(fc == null){
						throw new RuntimeException("First children of parameter "+pars[i].getIndex()+" not defined.");
					}
					//this parameter's value will be the context of its descendants
					newContext[depth][0] = pars[i].getPosition();
					newContext[depth][1] = pars[i].symbol;
					double temp = 1;
					//compute product over all first children, i.e. children that are the roots
					//of independent subtrees and whose context is fully defined by the current parameter
					for(int j=0;j<fc.length;j++){
						temp *= trees[fc[j]].getZ(newContext,trees);
					}
					//set the Z-part of the local normalization (for partial normalization constant)
					pars[i].setZ( temp );
					//for fullNormalizer multiplied by current value
					val += pars[i].getExpValue()*temp;
				}
				fullNormalizer = val;
				return fullNormalizer;
			}
		}

		private void invalidateNormalizers() {
			if(children != null){
				for(int i=0;i<children.length;i++){
					children[i].invalidateNormalizers();
				}
			}else{
				for(int i=0;i<pars.length;i++){
					pars[i].invalidateNormalizers();
					symT[i] = null;
				}
			}
			fullNormalizer = null;
		}


		public TreeElement clone() throws CloneNotSupportedException{
			TreeElement clone = (TreeElement) super.clone();
			if(children != null){
				clone.children = new TreeElement[children.length];
				for(int i=0;i<children.length;i++){
					clone.children[i] = children[i].clone();
				}
			}else{
				clone.children = null;
			}
			if(pars != null){
				clone.pars = new Parameter[pars.length];
				for(int i=0;i<pars.length;i++){
					clone.pars[i] = pars[i].clone();
				}
				clone.fullNormalizer = null;
				clone.symT = new Double[pars.length];
			}
			
			return clone;
		}

		

		private void divideByUnfree() {
			if(pars != null){
				double div = pars[pars.length - 1].getValue();
				for(int i=0;i<pars.length;i++){
					if(!Double.isNaN(pars[i].getValue() - div)&&!Double.isInfinite(pars[i].getValue() - div)){
						pars[i].setValue( pars[i].getValue() - div );
					}else{
						pars[i].setValue(0d);
					}
				}
			}else{
				for(int i=0;i<children.length;i++){
					children[i].divideByUnfree();
				}
			}
		}

		private LinkedList<Parameter> linearizeParameters(LinkedList<Parameter> list) {
			if(children != null){
				for(int i=0;i<children.length;i++){
					children[i].linearizeParameters(list);
				}
			}else{
				for(int i=0;i<pars.length;i++){
					list.add(pars[i]);
				}
			}
			return list;
		}

		private TreeElement(StringBuffer representation) throws NonParsableException{
			representation = XMLParser.extractForTag(representation, "treeElement");
			contextPos = XMLParser.extractIntForTag(representation, "contextPos");
			StringBuffer temp = XMLParser.extractForTag(representation, "children");
			if(temp.toString().equals("null")){
				children = null;
			}else{
				XMLParser.addTags(temp, "children");
				String[] childRep = XMLParser.extractStringArrayForTag(temp, "children");
				children = new TreeElement[childRep.length];
				for(int i=0;i<childRep.length;i++){
					children[i] = new TreeElement(new StringBuffer(childRep[i]));
				}
			}
			temp = XMLParser.extractForTag(representation, "pars");
			if(temp.toString().equals("null")){
				pars = null;
			}else{
				XMLParser.addTags(temp, "pars");
				pars = (Parameter[]) ArrayHandler.cast(XMLParser.extractStorableArrayForTag(temp, "pars"));
			}
			if(pars != null){
				symT = new Double[pars.length];
				fullNormalizer = null;
			}
			
		}
		
		private void setParameterFor(int depth, int symbol, int[][] context, Parameter par) {
			if(children!=null){
				//System.out.println("going to child for "+context[depth][1]+" in tree "+pos);
				for(int i=1;i<context[depth].length;i++){
					children[ context[depth][i] ].setParameterFor(depth + 1, symbol, context, par);
				}
			}else{
				//System.out.println("setting parameter for "+symbol);
				pars[symbol] = par;
			}
		}
		
		private void print(){
			System.out.println(contextPos);
			if(children != null){
				for(int i=0;i<children.length;i++){
					System.out.println("child "+i+":");
					children[i].print();
				}
			}else{
				for(int i=0;i<pars.length;i++){
					pars[i].print();
				}
			}
		}

		private void normalizePlugInParameters() {
			if(children != null){
				for(int i=0;i<children.length;i++){
					children[i].normalizePlugInParameters();
				}
			}else{
				double sum = 0;
				for(int i=0;i<pars.length;i++){
					sum += pars[i].getCounts();
				}
				if(sum > 0){
					for(int i=0;i<pars.length;i++){
						pars[i].setValue(Math.log(pars[i].getCounts()/sum));
					}
				}else{
					for(int i=0;i<pars.length;i++){
						pars[i].setValue(-Math.log(pars.length));
					}
				}
			}
		}

		private Parameter getParameterFor(Sequence seq, int start){
			if(children != null){
				return children[seq.discreteVal(contextPos+start)].getParameterFor(seq,start);
			}else{
				return pars[seq.discreteVal(pos+start)];
			}
		}

		/* (non-Javadoc)
		 * @see de.jstacs.Storable#toXML()
		 */
		public StringBuffer toXML() {
			StringBuffer source = new StringBuffer();
			XMLParser.appendIntWithTags(source, contextPos, "contextPos");
			if(children != null){
				XMLParser.appendStorableArrayWithTags(source, children, "children");
			}else{
				XMLParser.appendStringWithTags(source, "null", "children");
			}
			if(pars != null){
				XMLParser.appendStorableArrayWithTags(source, pars, "pars");
			}else{
				XMLParser.appendStringWithTags(source, "null", "pars");
			}
			XMLParser.addTags(source, "treeElement");
			return source;
		}

		private double getKLDivergence(double[] q){
			if(children != null){
				double val = 0;
				for(int i=0;i<children.length;i++){
					val += children[i].getKLDivergence(q);
				}
				return (val/(double)q.length);
			}else{
				double val = 0;
				double norm = 0;
				for(int i=0;i<pars.length;i++){
					norm += pars[i].getExpValue()*pars[i].getZ();
				}
				for(int i=0;i<pars.length;i++){
					double temp = pars[i].getExpValue()*pars[i].getZ()/norm;
					val += temp*Math.log(temp/q[i]);
				}
				return val;
			}
		}

		private void fill(double[] fillEmptyWith) {
			if(children != null){
				for(int i=0;i<children.length;i++){
					children[i].fill(fillEmptyWith);
				}
			}else{
				if(pars[pars.length - 1].isFree()){
					if(fillEmptyWith.length != pars.length){
						throw new IndexOutOfBoundsException("Different number of values ("+fillEmptyWith.length+") than free parameters ("+pars.length+").");
					}else{
						for(int i=0;i<pars.length;i++){
							pars[i].setValue( Math.log( fillEmptyWith[i] ) );
						}
					}
				}else{
					for(int i=0;i<pars.length-1;i++){
						pars[i].setValue( Math.log( fillEmptyWith[i] ) - Math.log( fillEmptyWith[pars.length - 1] ) );
					}
				}
			}
		}

		private void copy(TreeElement node) {
			if(this.children != null){
				if(node.children != null){
					if(this.children.length != node.children.length){
						throw new IndexOutOfBoundsException("Different number of children.");
					}else{
						for(int i=0;i<children.length;i++){
							this.children[i].copy(node.children[i]);
						}
					}
				}else{
					for(int i=0;i<children.length;i++){
						this.children[i].copy(node);
					}
				}
			}else{
				if(node.pars != null){
					if(this.pars.length != node.pars.length){
						throw new IndexOutOfBoundsException("Different number of parameters.");
					}else{
						for(int i=0;i<pars.length;i++){
							pars[i].setValue(node.pars[i].getValue());
						}
					}
				}else{
					for(int i=0;i<pars.length;i++){
						pars[i].setValue(node.sumFor(i));
					}
				}
			}
		}

		private double sumFor(int idx) {
			if(children != null){
				double sum = 0;
				for(int i=0;i<children.length;i++){
					sum += children[i].sumFor(idx);
				}
				return sum;
			}else{
				return pars[idx].getValue();
			}
		}

		private void initializeRandomly(double ess) {
			if(pars != null){
				
				if(ess <= 0){
					ess = alphabet.getAlphabetLengthAt( pars[0].getPosition() );
				}
				double[] hyp = new double[pars.length];
				for(int i=0;i<hyp.length;i++){
					hyp[i] = ess/alphabet.getAlphabetLengthAt( pars[i].getPosition() );
				}
				double[] temp = DirichletMRG.DEFAULT_INSTANCE.generate( pars.length, new DirichletMRGParams(hyp) );
				for(int i=0;i<pars.length;i++){
					pars[i].count = temp[i];
				}
				this.normalizePlugInParameters();
				if(!pars[pars.length - 1].isFree()){
					this.divideByUnfree();
				}
			}else{
				for(int i=0;i<children.length;i++){
					children[i].initializeRandomly(ess/alphabet.getAlphabetLengthAt( this.contextPos ));
				}
			}
			
		}

		private double computeGammaNorm() {
			if(children != null){
				double val = 0;
				for(int i=0;i<children.length;i++){
					val += children[i].computeGammaNorm();
				}
				return val;
			}else{
				double val = 0;
				double hypSum = 0, alpha;
				for(int i=0;i<pars.length;i++){
					alpha = pars[i].getPseudoCount();
					hypSum += alpha;
					val -= Gamma.logOfGamma( alpha );
				}
				val += Gamma.logOfGamma( hypSum );
				return val;
			}
		}
		
	}
	
}
