/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.io.XMLParser;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This scoring function does nothing. So it is possible to save parameters in an optimization.
 * 
 * @author Jens Keilwagen, Jan Grau
 */
public class UniformScoringFunction extends AbstractNormalizableScoringFunction
{
	private double ess, logP;
	
	/**
	 * The main constructor.
	 * 
	 * @param alphabets the alphabetContainer
	 * @param length the length of the modeled sequence
	 * @param ess the class ess
	 */
	public UniformScoringFunction( AlphabetContainer alphabets, int length, double ess )
	{
		super( alphabets, length );
		if( !alphabets.isDiscrete() )
		{
			throw new IllegalArgumentException( "The given AlphabetContainer has to be discrete." );
		}
		int l = alphabets.getPossibleLength();
		if( l != 0 && l != length )
		{
			throw new IllegalArgumentException( "The given AlphabetContainer (possible length = " + l + ") and length (" + length + ") do not match." );
		}
		if( ess < 0 )
		{
			throw new IllegalArgumentException( "The given ess has to be non-negative." );
		}
		this.ess = ess;
		computeLogP();
	}

	/**
	 * This is the constructor for {@link Storable}.
	 * 
	 * @param xml the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public UniformScoringFunction( StringBuffer xml ) throws NonParsableException
	{
		super( xml );
		computeLogP();
	}

	private void computeLogP()
	{
		logP = 1;
		for( int i = 0; i < length; i++ )
		{
			logP *= alphabets.getAlphabetLengthAt(i);
		}
		logP = -Math.log( logP );
	}
	
	public String getInstanceName()
	{
		return "uniform";
	}

	public double getLogScore( Sequence seq, int start )
	{
		return logP;
	}

	public double getLogScoreAndPartialDerivation( Sequence seq, int start, IntList indices, DoubleList dList )
	{
		return logP;
	}

	public int getNumberOfParameters()
	{
		return 0;
	}

	public void setParameters( double[] params, int start )
	{
	}

	public StringBuffer toXML()
	{
		StringBuffer b = new StringBuffer(1000);
		XMLParser.appendIntWithTags( b, length, "length" );
		XMLParser.appendStorableWithTags( b, alphabets, "alphabets" );
		XMLParser.appendDoubleWithTags( b, ess, "ess" );
		XMLParser.addTags( b, getClass().getSimpleName() );
		return b;		
	}

	public double getNormalizationConstant()
	{
		return 1;
	}

	public void initializeFunction( int index, boolean meila, Sample[] data, double[][] weights )
	{
		// does nothing
	}

	protected void fromXML( StringBuffer xml ) throws NonParsableException
	{
		StringBuffer b = XMLParser.extractForTag( xml, getClass().getSimpleName() );
		length = XMLParser.extractIntForTag( b, "length" );
		alphabets = (AlphabetContainer) XMLParser.extractStorableForTag( b, "alphabets" );
		ess = XMLParser.extractDoubleForTag( b, "ess" );
	}

	public int getSizeOfEventSpaceForRandomVariablesOfParameter(int index) 
	{
		return 0;
	}

	public double getPartialNormalizationConstant( int parameterIndex ) throws Exception
	{
		throw new IndexOutOfBoundsException( "Since a uniform scoring function has no parameters, this method can not be used" );
	}

	public double getEss()
	{
		return ess;
	}
	
	public String toString()
	{
		StringBuffer info = new StringBuffer(length*100);
		double val;
		for( int j = 0; j < length; j++ )
		{
			val = 1d / alphabets.getAlphabetLengthAt(0);
			info.append( j + "\t" + val + " for each element of " + alphabets.getAlphabetAt(j).toString() );
			if( j < length-1 )
			{
				info.append( "\n" );
			}
		}
		return info.toString();
	}
	
	public double getLogPriorTerm()
	{
		// since the normalization constant does not depend on any parameter,
		// it is constant and therefore left out
		return 0;
	}

	public void addGradientOfLogPriorTerm( double[] grad, int start ){}

	public double[] getCurrentParameterValues() throws Exception
	{
		return new double[0];
	}

	public boolean isInitialized()
	{
		return true;
	}

	public boolean isNormalized()
	{
		return true;
	}
	
	public void initializeFunctionRandomly( boolean freeParams ) throws Exception
	{	
	}
}
