/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import de.jstacs.Storable;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This interface is the main part of any ScoreClassifier.
 * 
 * @author Jens Keilwagen, Jan Grau
 */
public interface ScoringFunction extends Storable, Cloneable
{
	/**
	 * Indicates that the number of parameters of this {@link ScoringFunction} is not known (yet).
	 */
	public static final int UNKNOWN = -1;

	/**
	 * Creates a clone (deep copy) of the current <code>ScoringFunction</code> instance.
	 * @return the cloned instance
	 * @throws CloneNotSupportedException
	 */
	public ScoringFunction clone() throws CloneNotSupportedException;

	/**
	 * This method creates the underlying structure of the scoring function.
	 * 
	 * @param index
	 *            the index of the class the scoring function models
	 * @param freeParams
	 *            if <code>true<code>, the (reduced) parameterization is used
	 * @param data
	 *            the samples
	 * @param weights
	 *            the weights of the sequences in the samples
	 * @throws Exception
	 */
	public void initializeFunction( int index, boolean freeParams, Sample[] data, double[][] weights )
			throws Exception;

	/**
	 * This method initializes the scoring function randomly. It has to create the
	 * underlying structure of the scoring function.
	 * 
	 * @param freeParams if true, the (reduced) parameterization is used
	 * 
	 * @throws Exception
	 */
	public void initializeFunctionRandomly( boolean freeParams ) throws Exception;
	

	
	/**
	 * Returns the {@link AlphabetContainer} for this {@link ScoringFunction}.
	 * Only {@link Sequence}s with a comparable AlphabetContainer can be modeled.
	 *  
	 * @return the alphabet
	 */
	public AlphabetContainer getAlphabetContainer();

	/**
	 * Returns a <b>short</b> instance name.
	 * 
	 * @return a <b>short</b> instance name
	 */
	public String getInstanceName();

	/**
	 * Returns the length of this {@link ScoringFunction}. i.e. the length of the {@link Sequence}s this {@link ScoringFunction} can handle. For homogeneous {@link ScoringFunction}, i.e.
	 * {@link ScoringFunction}s that support {@link Sequence}s of different lengths, should return <code>0</code>.
	 * @return the length
	 */
	public int getLength();

	/**
	 * Returns the log score for the sequence
	 * 
	 * @param seq
	 *            the sequence
	 * 
	 * @return the log score for the sequence
	 */
	public double getLogScore( Sequence seq );

	/**
	 * Returns the log score for the sequence
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the startposition in the sequence
	 * 
	 * @return the log score for the sequence
	 */
	public double getLogScore( Sequence seq, int start );

	/**
	 * Returns the log score for the sequence and fills the list with the indices and the partial derivations.
	 * 
	 * @param seq
	 *            the sequence
	 * @param indices
	 *            after method invocation the list should contain the indices i where \frac{\partial \log
	 *            score(seq)}{\partial \lambda_i} is not zero
	 * @param partialDer
	 *            after method invocation the list should contain the corresponding \frac{\partial \log
	 *            score(seq)}{\partial \lambda_i}
	 * 
	 * @return the log score
	 */
	public double getLogScoreAndPartialDerivation( Sequence seq, IntList indices,
			DoubleList partialDer );

	/**
	 * Returns the log score for the sequence and fills the list with the indices and the partial derivations.
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the startposition in the sequence
	 * @param indices
	 *            after method invocation the list should contain the indices i where \frac{\partial \log
	 *            score(seq)}{\partial \lambda_i} is not zero
	 * @param partialDer
	 *            after method invocation the list should contain the corresponding \frac{\partial \log
	 *            score(seq)}{\partial \lambda_i}
	 * 
	 * @return the log score
	 */
	public double getLogScoreAndPartialDerivation( Sequence seq, int start, IntList indices,
			DoubleList partialDer );

	/**
	 * The number of parameters in this scoring function. If the number of parameters is not known yet, the method
	 * returns <code>UNKNOWN</code>.
	 * 
	 * @return the number of parameters in this scoring function
	 * 
	 * @see ScoringFunction#UNKNOWN
	 */
	public int getNumberOfParameters();

	/**
	 * This method return the number of recommended optimization starts. The standard implementation returns 1.
	 * 
	 * @return the number of recommended optimization starts
	 */
	public int getNumberOfRecommendedStarts();

	/**
	 * Returns a double array of dimension <code>getNumberOfParameters()</code> containing the current parameter
	 * values. If on e likes to use these parameters to start an optimization it is highly recommended to invoke
	 * {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])} before. After an optimization
	 * this method can be used to get the current parameter values.
	 * 
	 * @return the current parameter values
	 * 
	 * @throws Exception is thrown if no parameters exist, yet
	 */
	public double[] getCurrentParameterValues() throws Exception;

	/**
	 * This method sets the internal parameters to the values of <code>params</code> between <code>start</code> and
	 * <code>start + this.getNumberOfParameters() - 1</code>
	 * 
	 * @param params
	 *            the parameters
	 * @param start
	 *            the start index
	 */
	public void setParameters( double[] params, int start );

	/**
	 * This method can be used to determine whether the model is initialized. If the model is not initialize you should
	 * invoke the method {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])}.
	 * 
	 * @return <code>true</code> if the model is initialized
	 */
	public boolean isInitialized();
	
	/**
	 * Returns the initial class parameter for the class this {@link ScoringFunction} is responsible for, based on the probability <code>classProb</code>.
	 * @param classProb the class probability
	 * @return the class parameter
	 */
	public double getInitialClassParam( double classProb );
}
