/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.AlphabetContainerParameterSet;
import de.jstacs.parameters.InstanceParameterSet;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.NumberValidator;

/**
 * The super ParameterSet for any ParameterSet of a {@link DiscreteGraphicalModel}.
 * 
 * @author Jens Keilwagen
 */
public abstract class DGMParameterSet extends InstanceParameterSet
{
	/**
	 * The constructor for the {@link de.jstacs.Storable} interface.
	 * 
	 * @param s
	 *            the StringBuffer
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer is not parsable
	 */
	protected DGMParameterSet( StringBuffer s ) throws NonParsableException
	{
		super( s );
	}

	/**
	 * An empty constructor of extended classes.
	 * 
	 * @param instanceClass
	 *            the (sub-)class
	 * @param simple
	 *            whether the alphabet should be simple or not
	 * @param variableLength
	 * 			  whether the model can handle sequences of variable length
	 */
	protected DGMParameterSet( Class<? extends DiscreteGraphicalModel> instanceClass, boolean simple, boolean variableLength )
	{
		super( instanceClass, true, simple, variableLength );
	}

	/**
	 * The constructor for models, that can handle variable lengths.
	 * 
	 * @param instanceClass
	 *            the (sub-)class
	 * @param alphabet
	 *            the alphabet
	 * @param ess
	 *            the ESS
	 * @param description
	 *            the description
	 * 
	 * @throws Exception
	 *             if something went wrong
	 */
	protected DGMParameterSet( Class<? extends DiscreteGraphicalModel> instanceClass, AlphabetContainer alphabet,
			double ess, String description ) throws Exception
	{
		this( instanceClass, alphabet, 0, true, ess, description );
	}

	/**
	 * The constructor for models that can handle only sequences of fixed length.
	 * 
	 * @param instanceClass
	 *            the (sub-)class
	 * @param alphabet
	 *            the alphabet
	 * @param length
	 *            the length of the modeled sequences
	 * @param ess
	 *            the ESS
	 * @param description
	 *            the description
	 * 
	 * @throws Exception
	 *             if something went wrong
	 */
	protected DGMParameterSet( Class<? extends DiscreteGraphicalModel> instanceClass, AlphabetContainer alphabet,
			int length, double ess, String description ) throws Exception
	{
		this( instanceClass, alphabet, length, false, ess, description );
	}
			
	private DGMParameterSet( Class<? extends DiscreteGraphicalModel> instanceClass, AlphabetContainer alphabet,
			int length, boolean variableLength, double ess, String description ) throws Exception
	{
		super( instanceClass, alphabet, length, variableLength );
		loadParameters();
		setEss( ess );
		if( description != null )
		{
			parameters.get( 1 ).setValue( description );
		}
	}

	protected void loadParameters() throws Exception
	{
		initParameterList();
		parameters.add( new SimpleParameter( DataType.DOUBLE, "ESS", "the equivalent sample size", true,
				new NumberValidator<Double>( new Double( 0 ), new Double( Double.MAX_VALUE ) ) ) );
		parameters.add( new SimpleParameter( DataType.STRING, "description",
				"a textual description or comment for the model", false, "none" ) );
	}

	public boolean hasDefaultOrIsSet()
	{
		if( super.hasDefaultOrIsSet() )
		{
			// return getAlphabet().isDiscrete();
			return ((AlphabetContainerParameterSet) alphabet.getValue()).isDiscrete();
		}
		else
		{
			return false;
		}
	}

	public String getInstanceName()
	{
		return getClass().getSimpleName();
	}

	public DGMParameterSet clone() throws CloneNotSupportedException
	{
		return (DGMParameterSet) super.clone();
	}
	
	/**
	 * This method can be used to set the ess of this parameter set.
	 * 
	 * @param ess the ESS
	 * 
	 * @throws IllegalValueException if ess is negative
	 */
	public void setEss( double ess ) throws IllegalValueException
	{
		parameters.get( 0 ).setValue( ess );
	}
}
