/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.assessment;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.parameters.ExpandableParameterSet;
import de.jstacs.parameters.ParameterSet;
import de.jstacs.parameters.ParameterSetContainer;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.NumberValidator;
import de.jstacs.results.CategoricalResult;
import de.jstacs.results.NumericalResult;
import de.jstacs.results.Result;

/**
 * This class implements a <code>ClassifierAssessmentAssessParameterSet</code> that must be used
 * to call method <code>assess()</code> of a <code>RepatedSubSamplingExperiment</code>.
 * It contains user-specific parameters necessary for a run of a <code>RepeatedSubSamplingExperiment</code>.
 * 
 * @author andr|e gohr (a0coder (nospam:@) gmail (nospam:.) com)
 *
 */
public class RepeatedSubSamplingAssessParameterSet extends ClassifierAssessmentAssessParameterSet {

//	**********************
//	class variables
//	**********************

//	**********************
//	class methods
//	**********************

//	**********************
//	member variables
//	**********************

//	**********************
//	constructors
//	**********************
	
	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	protected RepeatedSubSamplingAssessParameterSet
	(Class alternativeInstanceClass) {
	super(alternativeInstanceClass);
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public RepeatedSubSamplingAssessParameterSet
	() throws UnsupportedOperationException {
	super();
	}
	
	
	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	protected RepeatedSubSamplingAssessParameterSet
	(StringBuffer representation) throws NonParsableException {
	super(representation);
	}

	/**
	 * 
	 * @param elementLength 	defines the length of elements (sequences) the classifiers
	 * 							to be assessed are able to classify
	 * 
	 * @param exceptionIfMPNotComputable a <code>RepeatedSubSamplingAssessParameterSet</code> is used
	 * 							in combination with an <code>MeasureParameters</code>-object
	 * 							to call <code>assess</code>-methods of <code>RepeatedSubSamplingExperiment</code>s.
	 * 						    If <code>exceptionIfMPNotComputable=true</code> then an expection is thrown
	 * 							in case of a selected measure-parameters that could not be computed.
	 * 
	 * @param repeats 			the number of repeates of each iteration 
	 * 							(subsample test- and train-data-sets from user-supplied data, 
	 * 							train classifiers using train-data-sets and test them using test-data-sets)
	 * 							of that <code>RepeatedHoldOutExperiment</code> this 
	 * 							<code>RepatedHoldOutAssessParameterSet</code> is used with
	 *  
	 * @param trainNumbers 		an array containing for each class (the classifiers to be assessed are
	 * 							capable to distinguish) the number of elements the subsampled
	 * 							train-data-sets should contain 
	 * 							
	 * @param testNumbers 		an array containing for each class (the classifiers to be assessed are
	 * 							capable to distinguish) the number of elements the subsampled
	 * 							test-data-sets should contain
	 * 
	 * @throws IllegalValueException is thrown in case of out-of-range or invalid given parameters
	 */
	public RepeatedSubSamplingAssessParameterSet
	(int elementLength, boolean exceptionIfMPNotComputable,
	int repeats ,int[] trainNumbers, int[] testNumbers) throws IllegalValueException {
	super(elementLength, exceptionIfMPNotComputable);
	
		this.parameters.get(2).setValue( new Integer(repeats) );
	
		ParameterSet[] tempPSA = new ParameterSet[trainNumbers.length];
		for(int i=0; i<tempPSA.length;			
			tempPSA[i]=getParameterSetContainingASingleIntValue(trainNumbers[i++],"training")
		);
	
		( (ExpandableParameterSet)(( (ParameterSetContainer)(this.parameters.get(3)) ).getValue()) ).replaceContentWith(tempPSA);

		ParameterSet[] tempPSA2 = new ParameterSet[testNumbers.length];
		for(int i=0; i<tempPSA2.length;			
			tempPSA2[i]=getParameterSetContainingASingleIntValue(testNumbers[i++],"testing")
		);
	
		( (ExpandableParameterSet)(( (ParameterSetContainer)(this.parameters.get(4)) ).getValue()) ).replaceContentWith(tempPSA2);
	
	}

	
//	**********************
//	member methods
//	**********************
	
	/**
	 * Creates a new ParameterSet containing a single Double-SimpleParameter.
	 * This ParameterSet is used as a part of the ExpandableParameterSet that contains
	 * the test-data-percent for a specific class. <br>
	 * @param percent the double-value to be contained in the returned ParameterSet.
	 * If percet=Double.NaN, no values is contained in the returned ParameterSet 
	 * (The SimpleParameter contained in the returned ParameterSet contains no value).
	 * @throws IllegalValueException 
	 */
	private ParameterSet getParameterSetContainingASingleIntValue
	(int num, final String train_test) 
	throws IllegalValueException{
		

		ParameterSet ret = new ParameterSet(){
   								protected void loadParameters
   								() throws Exception{
   								initParameterList(1);
   								this.parameters.add( new SimpleParameter(
   										DataType.INT,
   											"number",
   											"Defines a number of elements of data used as "+train_test+
   											" items (class-specific) during a SubSamplingAssessment",
   											true,
   											new NumberValidator<Integer>(1,Integer.MAX_VALUE))
   										);
   								}

   								public String getInstanceName
   								(){
   			                    return "";
   								}

   								public String getInstanceComment
   								(){
   								return "Contains only one Integer-object. This int-value " +
   								"defines the number of items that shoud be subsampled from the given data " +
   								"(for a specific class) and used as " + train_test + " items "+
   								"in a SubSamplingAssessment.";
   								}
			};
			
			ret.getParameterAt(0).setValue(new Integer(num));
			
	return ret;
	}
	
	
	@Override
	protected void initializeMyParametersArrayList
	(){
		initParameterList(6);
	}
	
	
	@Override
	protected void loadParameters
	() throws Exception {
	super.loadParameters();	
				
		//2-k
		this.parameters.add( new SimpleParameter(DataType.INT,
													"repeats",
													"Determines how often the procedure of " +
													"train/test classifers with random created " +
													"train- and test-data should be repeated.",
													true,
													new NumberValidator<Integer>(1,Integer.MAX_VALUE))
						    );
		
		
		
		//3-percents
		this.parameters.add( new ParameterSetContainer("trainDataNumbers",
													   "A RepeatedSubSamplingExperiment subsamples " +
													   "the used train- and test-datasets from the given data (for each class) " +
													   "in each iteration. This ParameterSetcontainer " +
													   "contains an ExpandableParameterSet that contains for each class " +
													   "the number of the items (for each class), that should be subsampled " +
													   "and used as training-data.",
													   new ExpandableParameterSet(null,
															   					  getParameterSetContainingASingleIntValue(1,"training"),
													   							  "number",
													   							  "At pos i in this Expandable ParameterSet " +
													   							  "defines the number of subsampled items, " +
													   							  "that should be used as train-data for class " +
													   							  "i in a RepeatedSubSamplingExperiment."
													   	)//new ExpandableParameterSet
											)//new ParameterSetContainer
							);//this.parameters.add(...)
		
		//4-percents
		this.parameters.add( new ParameterSetContainer("testDataNumbers",
													   "A RepeatedSubSamplingExperiment subsamples " +
													   "the used train- and test-datasets from the given data (for each class) " +
													   "in each iteration. This ParameterSetcontainer " +
													   "contains an ExpandableParameterSet that contains for each class " +
													   "the number of the items (for each class), that should be subsampled " +
													   "and used as test-data.",
													   new ExpandableParameterSet(null,
															   					  getParameterSetContainingASingleIntValue(1,"testing"),
													   							  "number",
													   							  "At pos i in this Expandable ParameterSet " +
													   							  "defines the number of subsampled items, " +
													   							  "that should be used as test-data for class " +
													   							  "i in a RepeatedSubSamplingExperiment."
													   	)//new ExpandableParameterSet
											)//new ParameterSetContainer
							);//this.parameters.add(...)
	}

	@Override
	public String getInstanceName() {
	return "RepeatedSubSamplingExperiment";
	}

	@Override
	public String getInstanceComment() {
	return "This SubSamplingAssessParameterSet may be used to pass parameters into " +
			"RepeatedSubSamplingExperiment's method assess().";
	}

	

	/**
	 * @return 	the repeates defined by this <code>RepatedSubSamplingAssessParameterSet</code>
	 * 			(repeats defines how many iterations (train and test classifiers) 
	 * 			of that <code>RepeatedSubSamplingExperiment</code>
	 * 			this <code>RepeatedSubSamplingAssessParameterSet</code> is used with are performed)
	 */
	public int getRepeats
	(){
	return ((Integer)(this.getParameterAt(2).getValue())).intValue();
	}
	
	
	/**
	 * @param train_case if true then (train | test)=train  else =test
	 
	 * @return 			an array class-wise containing the number of elements the subsampled
	 * 			  		(train | test)-data-sets should consist of.
	 */
	public int[] getTrain_TestNumbers
	(boolean train_case){
		
		int pos;
		if(train_case){
			pos=3;
		}else{
			pos=4;
		}
		
		ExpandableParameterSet tempEPS = (ExpandableParameterSet)(this.getParameterAt(pos).getValue());
		
		int[] ret = new int[tempEPS.getNumberOfParameters()];
		
		for(int i=0; i<ret.length;i++){
			//holy shit, that's realy unsexy
			ret[i]=((Integer)(((ParameterSet)(tempEPS.getParameterAt(i).getValue())).getParameterAt(0).getValue())).intValue();
		};
		
	return ret;
	}
	
	
	@Override
	public Collection<Result> getAnnotation()
	{
		ArrayList<Result> l = new ArrayList<Result>(3);
		l.add( new NumericalResult( "repeats", "The number of iterations", getRepeats() ) );
		l.add( new CategoricalResult( "number of elements for train", "The number of the items (for each class), that was subsampled and used as training-data.",
				Arrays.toString( getTrain_TestNumbers( true ) ) ) );
		l.add( new CategoricalResult( "number of elements for assessment", "The number of the items (for each class), that should be subsampled and used as assessment-data.",
				Arrays.toString( getTrain_TestNumbers( false  ) ) ) );
		return l;
	}

}
