/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.io.XMLParser;

/**
 * This class holds the confusion matrix of a classifier.
 * 
 * @author Jens Keilwagen
 */
public class ConfusionMatrix implements Storable
{
	/**
	 * The confusion matrix
	 */
	private int[][] matrix;

	private int all;
	
	private static final String XML_TAG = "confusion matrix";

	/**
	 * This constructor creates an instance with a given number of classes.
	 * 
	 * @param classes the number of classes
	 */
	public ConfusionMatrix( int classes )
	{
		matrix = new int[classes][classes];
		all = 0;
	}
	
	/**
	 * This is the constructor for {@link Storable}.
	 * 
	 * @param representation the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public ConfusionMatrix( StringBuffer representation ) throws NonParsableException
	{
		StringBuffer xml = XMLParser.extractForTag( representation, XML_TAG );
		all = XMLParser.extractIntForTag( xml, "all" );
		matrix = XMLParser.extractInt2ArrayForTag( xml, "matrix" );
	}

	/**
	 * This method updates the confusion matrix.
	 * 
	 * @param realClass the real class index
	 * @param predictedClass the predicted class index 
	 * 
	 */
	public void add( int realClass, int predictedClass )
	{
		matrix[predictedClass][realClass]++;
		all++;
	}

	/**
	 * This method returns the confusion matrix as a 2D-int-array.
	 * 
	 * @return the confusion matrix
	 */
	public int[][] getMatrix()
	{
		int[][] res = new int[matrix.length][];
		for( int i = 0; i < matrix.length; i++ )
		{
			res[i] = matrix[i].clone();
		}
		return res;
	}

	/**
	 * This method returns the misclassification rate.
	 * 
	 * @return the misclassifcation rate
	 */
	public double getMisclassificationRate()
	{
		double correct = 0;
		for( int i = 0; i < matrix.length; i++ )
		{
			correct += matrix[i][i];
		}
		return 1d - correct / all;
	}

	public StringBuffer toXML()
	{
		StringBuffer xml = new StringBuffer(1000);
		XMLParser.appendIntWithTags( xml, all, "all" );
		XMLParser.appendInt2ArrayWithTags( xml, matrix, "matrix" );
		XMLParser.addTags( xml, XML_TAG );
		return xml;
	}
}
