/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier;
import java.io.BufferedReader;
import java.io.FileReader;
import java.util.Arrays;

import de.jstacs.classifier.ScoreBasedPerformanceMeasureDefinitions.ThresholdMeasurePair;
import de.jstacs.utils.DoubleList;

/**
 * This class contains a main-method that allows to compute some performance measures in a very simple way.
 * 
 * @author Jens Keilwagen
 */
public class Classification {

	private static double[] getSortedValues( String fName ) throws Exception
	{
		BufferedReader r = new BufferedReader( new FileReader( fName ) );
		String line;
		DoubleList dlist = new DoubleList();
		while( (line = r.readLine()) != null )
		{
			dlist.add( Double.parseDouble(line) );
		}
		r.close();
		double[] res = dlist.toArray();
		Arrays.sort( res );
		return res;
	}

	/**
	 * This main method computes some performance measures for two given score files
	 * 
	 * @param args the name/path of the score files
	 * 	<ul>	
	 * 		<li> <code>args[0]</code> file name of score file for class 0
	 * 		<li> <code>args[1]</code> file name of score file for class 1
	 * </ul>
	 * 
	 * @throws Exception if the performance measures could not be computed
	 */
	public static void main( String[] args ) throws Exception {
		double[] fg = getSortedValues( args[0] );
		double[] bg = getSortedValues( args[1] );

		ThresholdMeasurePair tmp = ScoreBasedPerformanceMeasureDefinitions.getSensitivityForSpecificity( fg, bg, 0.999 );
		System.out.println( "Sn (with Sp=99.9%) =\t" + tmp.getMeasure() + "\tthreshold =\t" + tmp.getThreshold() );
		tmp = ScoreBasedPerformanceMeasureDefinitions.getFPRForSensitivity( fg, bg, 0.95 );
		System.out.println( "FPR (with Sn=95%) =\t" + tmp.getMeasure() + "\tthreshold =\t" + tmp.getThreshold() );
		tmp = ScoreBasedPerformanceMeasureDefinitions.getPPVForSensitivity( fg, bg, 0.95 );
		System.out.println( "PPV with Sn=95%\t" + tmp.getMeasure() + "\tthreshold =\t" + tmp.getThreshold() );
		System.out.println( "AUC_ROC\t" + ScoreBasedPerformanceMeasureDefinitions.getAUC_ROC( fg, bg, null ) );
		System.out.println( "AUC_PR\t" + ScoreBasedPerformanceMeasureDefinitions.getAUC_PR( fg, bg, null ) );
	}

}
