/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.utils;


/**
 * This class can be used for normalisation of any double array or a part of a double array.
 * 
 * @author Jens Keilwagen
 */
public class Normalisation
{
	/**
	 * Returns the log of the sum of values.
	 * 
	 * @param lnVal
	 *            the logs of the values, i.e. <code>lnVal[i] = Math.log( val[i] )</code>
	 * 
	 * @return \log(\sum_i val[i])
	 * 
	 */		
	public static double getLogSum( double... lnVal )
	{
		return getLogSum( 0, lnVal.length, lnVal );
	}
	
	/**
	 * Returns the log of the sum of values.
	 * 
	 * @param start the first index in lnVal considered for the sum
	 * @param end the index after the last index considered for the sum
	 * @param lnVal
	 *            the logs of the values, i.e. <code>lnVal[i] = Math.log( val[i] )</code>
	 * 
	 * @return \log(\sum_{i=start}^{end - 1} val[i])
	 * 
	 */		
	public static double getLogSum(int start, int end, double... lnVal )
	{
		/*
		//default
		double logSum = lnVal[0];
		if( lnVal.length > 1 )
		{
			for( int i = 1; i < lnVal.length; i++ )
			{
				logSum = getLogSumAB( logSum, lnVal[i] );
			}
		}
		return logSum;
		*/
		// same proceeding as in normalisation-methods
		
		double offset = lnVal[start], sum = 0;
		int i = start+1;
		for( ; i < end; i++ )
		{
			offset = Math.max( offset, lnVal[i] );
			
		}
		if( Double.isInfinite( offset ) )
		{
			return Double.NEGATIVE_INFINITY;
		}
		else
		{
			for( i = start; i < end; i++ )
			{
				sum += Math.exp( lnVal[i] - offset );
				
			}
			return offset + Math.log( sum );
		}
		/**/
	}

	/**
	 * log-sum-normalisation on <code>d</code>
	 * 
	 * @return the sum of <code>d</code>
	 */
	public static double logSumNormalisation( double[] d )
	{
		return logSumNormalisation( d, 0, d.length, d, 0 );
	}

	/**
	 * log-sum-normalisation on <code>d</code> between <code>startD</code> and <code>endD</code>
	 * 
	 * @return the sum of <code>d</code> between <code>startD</code> and <code>endD</code>
	 */
	public static double logSumNormalisation( double[] d, int startD, int endD )
	{
		return logSumNormalisation( d, startD, endD, d, startD );
	}

	/**
	 * log-sum-normalisation on <code>d</code> between <code>startD</code> and <code>endD</code>
	 * 
	 * @return the sum of <code>d</code> between <code>startD</code> and <code>endD</code> and
	 *         <code>secondValue</code>
	 */
	public static double logSumNormalisation( double[] d, int startD, int endD, double[] secondValues )
	{
		return logSumNormalisation( d, startD, endD, secondValues, d, startD );
	}

	/**
	 * log-sum-normalisation on <code>d</code> between <code>startD</code> and <code>endD</code>, writing the
	 * result in <code>dest</code> starting at position <code>startDest</code>
	 * 
	 * @return the sum of <code>d</code> between <code>startD</code> and <code>endD</code>
	 */
	public static double logSumNormalisation( double[] d, int startD, int endD, double[] dest, int startDest )
	{
		return logSumNormalisation( d, startD, endD, null, dest, startDest );
	}

	/**
	 * log-sum-normalisation on <code>d</code> between <code>startD</code> and <code>endD</code>, writing the
	 * result in <code>dest</code> starting at position <code>startDest</code>
	 * 
	 * @return the sum of <code>d</code> between <code>startD</code> and <code>endD</code>
	 */
	public static double logSumNormalisation( double[] d, int startD, int endD, double[] secondValues, double[] dest,
			int startDest )
	{
		double offset = Double.NEGATIVE_INFINITY;
		int i = startD;
		for( ; i < endD; i++ )
		{
			offset = Math.max( offset, d[i] );
		}
		if( secondValues != null )
		{
			for( i = 0; i < secondValues.length; i++ )
			{
				offset = Math.max( offset, secondValues[i] );
			}
		}
		return logSumNormalisation( d, startD, endD, offset, secondValues, dest, startDest );
	}

	/**
	 * log-sum-normalisation on <code>d</code> using offset <code>offset</code>
	 * 
	 * @return the sum of <code>d</code>
	 */
	public static double logSumNormalisation( double[] d, double offset )
	{
		return logSumNormalisation( d, 0, d.length, offset, d, 0 );
	}

	/**
	 * log-sum-normalisation on <code>d</code> between <code>startD</code> and <code>endD</code>, writing the
	 * result in <code>dest</code> starting at position <code>startDest</code> using offset <code>offset</code>
	 * 
	 * @return the sum of <code>d</code> between <code>startD</code> and <code>endD</code>
	 */
	public static double logSumNormalisation( double[] d, int startD, int endD, double offset, double[] dest, int start )
	{
		return logSumNormalisation( d, startD, endD, offset, null, dest, start );
	}

	/**
	 * log-sum-normalisation on <code>d</code>, writing the result in <code>dest</code> starting at position
	 * </code>start<code> using offset <code>offset</code>
	 * 
	 * @return the sum of <code>d</code>
	 */
	public static double logSumNormalisation( double[] d, int startD, int endD, double offset, double[] secondValues,
			double[] dest, int startDest )
	{
		double sum = 0;
		int i = 0, l = endD - startD;
		for( ; i < l; i++ )
		{
			dest[startDest + i] = Math.exp( d[startD + i] - offset );
			sum += dest[startDest + i];
		}
		if( secondValues != null )
		{
			for( i = 0; i < secondValues.length; i++ )
			{
				secondValues[i] = Math.exp( secondValues[i] - offset );
				sum += secondValues[i];
			}
		}
		if( sum != 1d )
		{
			normalisation( dest, sum, startDest, startDest + l );
			if( secondValues != null )
			{
				normalisation( secondValues, sum );
			}
		}
		return offset + Math.log( sum );
	}

	/**
	 * sum normailsation on <code>d</code>
	 * 
	 * @return the sum of d
	 */
	public static double sumNormalisation( double[] d )
	{
		return sumNormalisation( d, d, 0 );
	}

	/**
	 * sum normalisation on <code>d</code>, writing the result in <code>dest</code> starting at position </code>start<code>
	 * 
	 * @return the sum of d
	 */
	public static double sumNormalisation( double[] d, double[] dest, int start )
	{
		int i;
		double sum = d[0];
		for( i = 1; i < d.length; i++ )
		{
			sum += d[i];
		}
		normalisation( d, sum, dest, start );
		return sum;
	}

	/**
	 * normalisation on <code>d</code>, using the value <code>v</code> to normalize
	 */
	public static void normalisation( double[] d, double v )
	{
		normalisation( d, v, d, 0 );
	}

	/**
	 * sum normalisation on <code>d</code>, writing the result in <code>dest</code> starting at position </code>start<code>
	 * using the value <code>v</code> to normalize
	 */
	public static void normalisation( double[] d, double v, double[] dest, int start )
	{
		for( int i = 0; i < d.length; i++, start++ )
		{
			dest[start] = d[i] / v;
		}
	}

	/**
	 * sum normalisation on <code>d</code>, starting at position </code>start<code>, stopping at position
	 * <code>end</code>, using the value <code>v</code> to normalize
	 */
	public static void normalisation( double[] d, double v, int start, int end )
	{
		for( ; start < end; start++ )
		{
			d[start] /= v;
		}
	}
}
