/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.btMeasures;

import de.jstacs.NonParsableException;
import de.jstacs.algorithms.graphs.MST;
import de.jstacs.data.Sample;
import de.jstacs.io.XMLParser;
import de.jstacs.scoringFunctions.directedGraphicalModels.BayesianNetworkScoringFunction;
import de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure;

/**
 * Structure learning {@link Measure} that computes a maximum spanning tree based on the explaining away residual and uses the resulting
 * tree structure as structure of a Bayesian tree (special case of a Bayesian network) in a {@link BayesianNetworkScoringFunction}.
 * 
 * @author Jan Grau
 *
 */
public class BTExplainingAwayResidual extends Measure {


	private double[] ess;
	
	/**
	 * Creates a new explaining away residual Bayesian tree {@link Measure}.
	 * @param ess the equivalent sample sizes for the classes
	 */
	public BTExplainingAwayResidual(double[] ess) {
		this.ess = ess;
	}

	/**
	 * Re-creates a {@link BTExplainingAwayResidual} from is XML-representation as returned by {@link BTExplainingAwayResidual#toXML()}.
	 * @param buf the XML-representation
	 * @throws NonParsableException is thrown if the XML-code could not be parsed
	 */
	public BTExplainingAwayResidual(StringBuffer buf) throws NonParsableException{
		buf = XMLParser.extractForTag(buf, "btExplainingAwayResidual");
		ess = XMLParser.extractDoubleArrayForTag(buf, "ess");
	}
	
	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#clone()
	 */
	public BTExplainingAwayResidual clone() throws CloneNotSupportedException{
		BTExplainingAwayResidual clone = (BTExplainingAwayResidual) super.clone();
		clone.ess = ess.clone();
		return clone;
	}
	
	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#getInstanceName()
	 */
	@Override
	public String getInstanceName() {
		return "Bayesian tree with explaining away residual";
	}

	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#getParents(de.jstacs.data.Sample, de.jstacs.data.Sample, double[], double[], int)
	 */
	@Override
	public int[][] getParents(Sample fg, Sample bg, double[] weightsFg, double[] weightsBg, int length) throws Exception {

		double[][][][] statFg = getStatistics(fg, weightsFg, length,ess[0]);
		double[][][][] statBg = getStatistics(bg, weightsBg, length, ess[1]);
		double nFg = sum(weightsFg) + ess[0];
		double nBg = sum(weightsBg) + ess[1];
		
		double[][] ear = getEAR(statFg, statBg, nFg, nBg);
		
		
		int[][] p = MST.kruskal( ear );
		
		int[][] parents = new int[length][1];
		for(int i=0;i<parents.length;i++){
			parents[i][0] = i;
		}
		for(int i=0;i<p.length;i++){
			int idx = p[i][1];
			parents[idx] = new int[2];
			parents[idx][0] = p[i][0];
			parents[idx][1] = idx;
			
		}
		return parents;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.Storable#toXML()
	 */
	public StringBuffer toXML() {
		StringBuffer buf = new StringBuffer();
		XMLParser.appendDoubleArrayWithTags(buf, ess, "ess");
		XMLParser.addTags(buf, "btExplainingAwayResidual");
		return buf;
	}

}
