/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import java.util.Random;

import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This class is the main part of any ScoreClassifier. It implements many methods of the interface {@link NormalizableScoringFunction}.
 * 
 * @author Jens Keilwagen, Jan Grau
 */
public abstract class AbstractNormalizableScoringFunction implements NormalizableScoringFunction
{
	/**
	 * This object can be used for drawing initial parameters.
	 * 
	 * @see AbstractNormalizableScoringFunction#getCurrentParameterValues()
	 */
	protected static final Random r = new Random();

	/**
	 * The AlphabetContainer of this {@link NormalizableScoringFunction}
	 */
	protected AlphabetContainer alphabets;

	/**
	 * The length of the modeled sequences.
	 */
	protected int length;

	/**
	 * The main constructor.
	 * 
	 * @param alphabets the AlphabetContainer of this ScoringFunction
	 * @param length the length of this ScoringFunction
	 */
	public AbstractNormalizableScoringFunction( AlphabetContainer alphabets, int length )
	{
		this.alphabets = alphabets;
		this.length = length;
	}

	/**
	 * This is the constructor for {@link de.jstacs.Storable}.
	 * 
	 * @param xml the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public AbstractNormalizableScoringFunction( StringBuffer xml ) throws NonParsableException
	{
		alphabets = null;
		length = -1;
		fromXML( xml );
		if( alphabets == null )
		{
			throw new NonParsableException( "AlphabetContainer could not be parsed." );
		}
		if( length < 0 || alphabets == null )
		{
			throw new NonParsableException( "Length could not be parsed." );
		}		
	}

	public AbstractNormalizableScoringFunction clone() throws CloneNotSupportedException
	{
		return (AbstractNormalizableScoringFunction) super.clone();
	}
	
	/**
	 * This method is called in the constructor to create a scoring function from a StringBuffer
	 * 
	 * @param xml
	 *            the XML representation
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer could not be parsed.
	 */
	protected abstract void fromXML( StringBuffer xml ) throws NonParsableException;

	public final AlphabetContainer getAlphabetContainer()
	{
		return alphabets;
	}

	public int getLength()
	{
		return length;
	}

	/**
	 * Returns the log score for the sequence
	 * 
	 * @param seq
	 *            the sequence
	 * 
	 * @return the log score for the sequence
	 */
	public final double getLogScore( Sequence seq )
	{
		return getLogScore( seq, 0 );
	}

	public final double getLogScoreAndPartialDerivation( Sequence seq, IntList indices,
			DoubleList partialDer )
	{
		return getLogScoreAndPartialDerivation( seq, 0, indices, partialDer );
	}

	public int getNumberOfRecommendedStarts()
	{
		return 1;
	}

	public boolean isNormalized()
	{
		return false;
	}
	
	/**
	 * This method checks whether all given NormalizableScoringFunctions are normalized.
	 * 
	 * @param function the NormalizableScoringFunctions to be checked
	 * 
	 * @return <code>true</code> if all NormalizableScoringFunctions are already normalized, otherwise <code>false</code>
	 * 
	 * @see NormalizableScoringFunction#isNormalized()
	 */
	public static boolean isNormalized( NormalizableScoringFunction... function )
	{
		int i = 0;
		while( i < function.length && function[i].isNormalized() )
		{
			i++;
		}
		return i == function.length;
	}
	
	public double getInitialClassParam( double classProb )
	{
		return Math.log( classProb ) - Math.log( getNormalizationConstant() );
	}
}
