/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete.inhomogeneous;

import java.util.Arrays;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;

/**
 * The constraint can be used for any MEM application.
 * 
 * @author Jens Keilwagen
 */
public class MEMConstraint extends InhConstraint
{
	private double[] expLambda;
	private double[] lambda;

	private int[] corrected_positions;

	private static int[] isSorted( int[] pos ) throws IllegalArgumentException
	{
		int i = 1;
		while( i < pos.length && pos[i - 1] < pos[i] )
		{
			i++;
		}
		if( i < pos.length )
		{
			throw new IllegalArgumentException( "The position array is not unique." );
		}
		return pos;
	}

	/**
	 * Creates a MEMConstraint as part of a (whole) model
	 * 
	 * @param pos
	 *            the used positions (have to be sorted)
	 * @param alphabetLength
	 *            an array containing the length of the the alphabet for each position (of the whole model)
	 * 
	 * @throws IllegalArgumentException
	 *             if <code>pos</code> is not sorted
	 */
	public MEMConstraint( int[] pos, int[] alphabetLength ) throws IllegalArgumentException
	{
		super( isSorted( pos ), alphabetLength );
		expLambda = new double[counts.length];
		Arrays.fill( expLambda,1);
		lambda = new double[counts.length];
		corrected_positions = usedPositions;
	}

	/**
	 * Creates a MEMConstraint as part of a model
	 * 
	 * @param pos
	 *            the used positions (have to be sorted)
	 * @param alphabetLength
	 *            an array containing the length of the the alphabet for each position (of the whole model)
	 * @param corrected_positions
	 *            an array containing the corrected positions
	 * 
	 * @throws IllegalArgumentException
	 *             if <code>pos</code> is not sorted or <code>pos.length!=corrected_positions.length</code>
	 */
	public MEMConstraint( int[] pos, int[] alphabetLength, int[] corrected_positions )
			throws IllegalArgumentException
	{
		super( isSorted( pos ), alphabetLength );
		if( pos.length != corrected_positions.length )
		{
			throw new IllegalArgumentException( "The length of pos and corrected_positions is not equal." );
		}
		expLambda = new double[counts.length];
		lambda = new double[counts.length];
		this.corrected_positions = corrected_positions.clone();
	}

	/**
	 * Creates a MEMConstraint from a StringBuffer
	 * 
	 * @param xml
	 *            the buffer
	 * 
	 * @throws NonParsableException
	 *             if the buffer is not parsable
	 */
	public MEMConstraint( StringBuffer xml ) throws NonParsableException
	{
		super( xml );
	}

	public MEMConstraint clone() throws CloneNotSupportedException
	{
		MEMConstraint clone = (MEMConstraint) super.clone();
		clone.expLambda = expLambda.clone();
		clone.lambda = lambda.clone();
		if( corrected_positions == usedPositions )
		{
			clone.corrected_positions = clone.usedPositions;
		}
		else
		{
			clone.corrected_positions = corrected_positions.clone();
		}
		return clone;
	}

	@Override
	public void estimate( double ess )
	{
		estimateUnConditional( 0, freq.length, ess / (double) freq.length, true );
	}

	/**
	 * Returns the value of the corrected position
	 * 
	 * @param index
	 * 
	 * @return the value of the corrected position
	 */
	public int getCorrectedPosition( int index )
	{
		return corrected_positions[index];
	}

	/**
	 * Returns \exp(\lambda_{index}).
	 * 
	 * @param index
	 *            the index
	 * 
	 * @return \exp(\lambda_{index})
	 */
	public double getExpLambda( int index )
	{
		return expLambda[index];
	}
	
	/**
	 * Returns \lambda_{index}.
	 * 
	 * @param index
	 *            the index
	 * 
	 * @return \lambda_{index}
	 */
	public double getLambda( int index )
	{
		return lambda[index];
	}

	/**
	 * Multiplies \exp(\lambda_{index}) with <code>val</code>
	 * 
	 * @param index
	 *            the index
	 * @param val
	 *            the factor/value
	 */
	public void multiplyExpLambdaWith( int index, double val )
	{
		expLambda[index] *= val;
		lambda[index] += Math.log( val );
	}

	public void reset()
	{
		super.reset();
		Arrays.fill( expLambda, 1d );
		Arrays.fill( lambda, 0d );
	}

	/**
	 * Returns the index of that constraint that is satiesfied by <code>sequence</code>
	 * 
	 * @param sequence
	 *            the SequenceIterator
	 * 
	 * @return the index of the fulfilled constraint
	 */
	public int satisfiesSpecificConstraint( SequenceIterator sequence )
	{
		int erg = 0, i = 0;
		for( ; i < corrected_positions.length; i++ )
		{
			//erg += offset[i] * sequence.getValueAt( corrected_positions[i] );
			erg += offset[i] * sequence.seq[ corrected_positions[i] ];
		}
		return erg;
	}

	public double getFreq( int index )
	{
		return freq[index];
	}
	
	/**
	 * Sets the value of exp(\lambda_{index})
	 * 
	 * @param index the index
	 * @param val the value
	 */
	public void setExpLambda( int index, double val )
	{
		expLambda[index] = val;
		lambda[index] = Math.log( val );
	}
	
	/**
	 * Sets the value of \lambda_{index}
	 * 
	 * @param index the index
	 * @param val the value
	 */
	public void setLambda( int index, double val )
	{
		expLambda[index] = Math.exp( val );
		lambda[index] = val;
	}

	@Override
	public String toString()
	{
		String erg = "" + usedPositions[0];
		for( int i = 1; i < usedPositions.length; i++ )
		{
			erg += ", " + usedPositions[i];
		}
		return erg;
	}

	protected void appendAdditionalInfo( StringBuffer xml )
	{
		super.appendAdditionalInfo( xml );
		XMLParser.appendDoubleArrayWithTags( xml, lambda, "lambda" );
		if( corrected_positions != usedPositions )
		{
			StringBuffer b = new StringBuffer( 500 );
			XMLParser.appendIntArrayWithTags( b, corrected_positions, "corrected positions" );
			XMLParser.addTags( b, "corrected" );
			xml.append( b );
		}
	}

	private static String XML_TAG = "MEMConstraint";

	protected String getXMLTag()
	{
		return XML_TAG;
	}

	protected void extractAdditionalInfo( StringBuffer xml ) throws NonParsableException
	{
		super.extractAdditionalInfo( xml );
		lambda = XMLParser.extractDoubleArrayForTag( xml, "lambda" );
		expLambda = new double[lambda.length];
		for( int i = 0; i < lambda.length; i++)
		{
			expLambda[i] = Math.exp( lambda[i] );
		}
		StringBuffer corrected = XMLParser.extractForTag( xml, "corrected" );
		if( corrected == null )
		{
			corrected_positions = usedPositions;
		}
		else
		{
			corrected_positions = XMLParser.extractIntArrayForTag( corrected, "corrected positions" );
		}
	}
}