/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete;

import java.util.Arrays;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.data.Sequence;
import de.jstacs.io.XMLParser;

/**
 * The main class for all constraints on models. This class centralize all specific constraint from a partitioning
 * constraint.
 * 
 * @author Jens Keilwagen
 */
public abstract class Constraint implements Storable
{
	/**
	 * The counts for each specific constraint
	 */
	protected double[] counts;

	/**
	 * The frequencies estimated from the counts
	 */
	protected double[] freq;

	/**
	 * The used positions
	 */
	protected int[] usedPositions;

	/**
	 * The main constructor. Checks that each position is used maximally once.
	 * 
	 * @param usedPositions
	 *            the used positions (will be cloned), hva to be non-negative
	 * @param n
	 *            the number of specific constraints
	 */
	public Constraint( int[] usedPositions, int n )
	{
		// is unique??
		int max = -1, i = 0;
		for( ; i < usedPositions.length; i++ )
		{
			if( usedPositions[i] < 0 )
			{
				throw new IllegalArgumentException( "The positions have to be non-negative." );
			}
			if( max < usedPositions[i] )
			{
				max = usedPositions[i];
			}
		}
		boolean[] used = new boolean[max + 1];
		for( i = 0; i < usedPositions.length; i++ )
		{
			if( used[usedPositions[i]] )
			{
				throw new IllegalArgumentException( "Each position can be used only once, corrupted at "
						+ usedPositions[i] + "." );
			}
			else
			{
				used[usedPositions[i]] = true;
			}
		}
		this.usedPositions = usedPositions.clone();
		counts = new double[n];
		freq = new double[n];
	}

	/**
	 * The constructor for creating an object from an StringBuffer. The StringBuffer contains the constraint in
	 * XML-format.
	 * 
	 * @param xml
	 *            the StringBuffer
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer is not parseable
	 */
	public Constraint( StringBuffer xml ) throws NonParsableException
	{
		StringBuffer erg = XMLParser.extractForTag( xml, getXMLTag() );
		usedPositions = XMLParser.extractIntArrayForTag( erg, "usedPositions" );
		counts = XMLParser.extractDoubleArrayForTag( erg, "counts" );
		freq = XMLParser.extractDoubleArrayForTag( erg, "freq" );
		extractAdditionalInfo( erg );
	}

	/**
	 * Adds the <code>weight</code> to the count with index <code>index</code>.
	 * 
	 * @param index
	 *            the index
	 * @param weight
	 *            the weight
	 */
	public void add( int index, double weight )
	{
		counts[index] += weight;
	}

	/**
	 * This method determines the specific constraint that is fulfilled by the sequence and adds the <code>weight</code>
	 * to the specific counter.
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the start position
	 * @param weight
	 *            the weight for the sequence
	 */
	public void add( Sequence seq, int start, double weight )
	{
		add( satisfiesSpecificConstraint( seq, start ), weight );
	}

	protected Constraint clone() throws CloneNotSupportedException
	{
		Constraint p = (Constraint) super.clone();
		p.counts = counts.clone();
		p.freq = freq.clone();
		p.usedPositions = usedPositions.clone();
		return p;
	}

	/**
	 * Estimates the (smoothed) relative frequencies using the ess.
	 * 
	 * @param ess
	 *            the ESS
	 */
	public abstract void estimate( double ess );

	/**
	 * Estimates unconditionally.
	 * 
	 * @param start
	 *            the start index
	 * @param end
	 *            the end index
	 * @param pc
	 *            the pseudocount for each parameter
	 * @param exceptionWhenNoData
	 *            if <code>true</code> an (runtime) exception is thrown if no data was available to estimate the
	 *            parameters
	 */
	protected void estimateUnConditional( int start, int end, double pc, boolean exceptionWhenNoData )
	{
		double sum = 0;
		for( int index = start; index < end; index++ )
		{
			sum += counts[index];
		}
		sum += (end - start) * pc;
		if( sum <= 0 )
		{
			if( exceptionWhenNoData )
			{
				throw new IllegalArgumentException( "A marginal distribution returned an illegal value (" + sum + ")." );
			}
			else
			{
				Arrays.fill( freq, start, end, 1d / (double) (end - start) );
			}
		}
		else
		{
			while( start < end )
			{
				freq[start] = (counts[start++] + pc) / sum;
			}
		}
	}

	/**
	 * This method appends additional information that are not stored in the base class to the StringBuffer.
	 * 
	 * @param xml
	 *            the StringBuffer that is used for appending additional information
	 */
	protected abstract void appendAdditionalInfo( StringBuffer xml );

	/**
	 * Returns the current count with index <code>index</code>
	 * 
	 * @param index
	 *            the index
	 * 
	 * @return the current count
	 */
	public double getCount( int index )
	{
		return counts[index];
	}

	/**
	 * Returns the current frequency with index <code>index</code>
	 * 
	 * @param index
	 *            the index
	 * 
	 * @return the current frequency
	 */
	public double getFreq( int index )
	{
		return freq[index];
	}

	/**
	 * This method determines the specific constraint that is fullfilled by the sequence beginning at <code>start</code>.
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the start position
	 * 
	 * @return the according frequency
	 */
	public double getFreq( Sequence seq, int start )
	{
		return getFreq( satisfiesSpecificConstraint( seq, start ) );
	}

	/**
	 * Returns the marginal order i.e. the number of used random variables.
	 * 
	 * @return the marginal order
	 */
	public int getMarginalOrder()
	{
		return usedPositions.length;
	}

	/**
	 * Returns the number of specific constraint.
	 * 
	 * @return the number of specific constraint
	 */
	public int getNumberOfSpecificConstraints()
	{
		return counts.length;
	}

	/**
	 * Returns the position with index <code>index</code>
	 * 
	 * @param index
	 *            the index
	 * @return the position with index <code>index</code>
	 */
	public int getPosition( int index )
	{
		return usedPositions[index];
	}

	/**
	 * Returns a clone of array of used positions.
	 * 
	 * @return a clone of array of used positions
	 */
	public int[] getPositions()
	{
		return usedPositions.clone();
	}

	/**
	 * Returns the XML-tag that is used for the class to en- or decode.
	 * 
	 * @return the XML-tag that is used for the class to en- or decode
	 */
	protected abstract String getXMLTag();

	/**
	 * This method resets all member variables the are in someway counters, frequencies, ...
	 */
	public void reset()
	{
		Arrays.fill( counts, 0 );
		Arrays.fill( freq, 0 );
	}

	/**
	 * This method returns the index of the specific constraint that is fullfilled by the sequence beginning at
	 * <code>start</code>.
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the start position
	 * 
	 * @return the index of the fulfilled, specific constraint
	 */
	public abstract int satisfiesSpecificConstraint( Sequence seq, int start );

	/**
	 * This method parses additional information from the StringBuffer that are not parsed in the base class.
	 * 
	 * @param xml
	 *            the StringBuffer to parse
	 * @throws NonParsableException
	 *             if something with the parsing went wrong
	 */
	protected abstract void extractAdditionalInfo( StringBuffer xml ) throws NonParsableException;

	public abstract String toString();

	public StringBuffer toXML()
	{
		StringBuffer erg = new StringBuffer( 10000 );
		XMLParser.appendIntArrayWithTags( erg, usedPositions, "usedPositions" );
		XMLParser.appendDoubleArrayWithTags( erg, counts, "counts" );
		XMLParser.appendDoubleArrayWithTags( erg, freq, "freq" );
		appendAdditionalInfo( erg );
		XMLParser.addTags( erg, getXMLTag() );
		return erg;
	}
}
