/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import java.io.File;
import java.io.FilenameFilter;

import javax.swing.filechooser.FileFilter;

/**
 * A simple filter on files.
 * 
 * @author Jens Keilwagen
 */
public class SubstringFilenameFilter extends FileFilter implements java.io.FileFilter, FilenameFilter {

	/**
	 * This enum defines the different types a string can be part of a other string.
	 * 
	 * @author Jens Keilwagen
	 */
	public static enum PartOfName {
		/**
		 * The string shall be prefix.
		 */
		PREFIX,
		/**
		 * The String shall be infix.
		 */
		INFIX,
		/**
		 * The String shall be suffix.
		 */
		SUFFIX
	}

	private PartOfName type;

	private String desc;

	private String[] substring;

	private boolean dir, ignoreCase;

	/**
	 * A simple constructor.
	 * 
	 * @param type
	 *            the type of the substring the will be searched
	 * @param desc
	 *            the description of the filter, e.g. &quot;text-files
	 *            (*.txt)&quot;
	 * @param dir
	 *            a switch allow directories to be accepted;
	 * @param ignoreCase
	 *            a switch whether to ignore the case of the file names or not
	 * @param substring
	 *            an array of substrings, at least one substring has to be found
	 *            as specific part (type) of the file name that a
	 *            &quot;real&quot; file will be accepted
	 */
	public SubstringFilenameFilter( PartOfName type, String desc, boolean dir, boolean ignoreCase, String... substring ) {
		this.type = type;
		this.desc = desc;
		this.dir = dir;
		this.ignoreCase = ignoreCase;
		this.substring = new String[substring.length];
		for( int i = 0; i < substring.length; i++ ) {
			this.substring[i] = ignoreCase ? substring[i].toLowerCase() : substring[i];
		}
	}

	public boolean accept( File arg0 ) {
		if( dir && arg0.isDirectory() ) {
			return true;
		}
		return accept( arg0.getParentFile(), arg0.getName() );
	}
	
	public boolean accept( File dir, String name ) {
		int i = 0;
		if( ignoreCase ) {
			name = name.toLowerCase();
		}
		switch( type ) {
			case PREFIX:
				while( i < substring.length && !name.startsWith( substring[i] ) ) {
					i++;
				}
				break;
			case INFIX:
				while( i < substring.length && !name.contains( substring[i] ) ) {
					i++;
				}
				break;
			case SUFFIX:
				while( i < substring.length && !name.endsWith( substring[i] ) ) {
					i++;
				}
				break;
		}
		return i < substring.length;
	}

	public String getDescription() {
		return desc;
	}
}
