/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data;

import de.jstacs.InstantiableFromParameterSet;
import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.parameters.ParameterSet;

/**
 * Class for an Alphabet. All Alphabets are immutable.
 * 
 * @author Jens Keilwagen
 */
public abstract class Alphabet implements Storable, InstantiableFromParameterSet, Comparable<Alphabet>
{
	/**
	 * Checks if two alphabets are consistent, i.e. both alphabets use the same symbols.
	 * 
	 * @param a
	 *            the second alphabet
	 * @return true if both alphabets are consistent, false otherwise
	 */
	public final boolean checkConsistency( Alphabet a )
	{
		return compareTo( a ) == 0;
	}

	/**
	 * Returns the minimal value.
	 * 
	 * @return the minimal value
	 */
	public abstract double getMin();

	/**
	 * Returns the length of the alphabet.
	 * 
	 * @return the length of the alphabet
	 */
	public abstract double length();

	public abstract String toString();
	
	/**
	 * The super class for the <code>ParameterSet</code> of any <code>Alphabet</code>.
	 * 
	 * @author Jens Keilwagen
	 */
	public static abstract class AlphabetParameterSet extends ParameterSet
	{
		/**
		 * Creates a new <code>AlphabetParameterSet</code>.
		 * 
		 * @param instanceClass the class
		 */
		public AlphabetParameterSet( Class<? extends Alphabet> instanceClass )
		{
			super( instanceClass );
		}

		/**
		 * Creates a new <code>AlphabetParameterSet</code> from its XML-representation.
		 * 
		 * @param representation
		 *            the XML-representation
		 * @throws NonParsableException
		 *             is thrown if <code>representation</code> could not be parsed
		 */
		public AlphabetParameterSet( StringBuffer representation ) throws NonParsableException
		{
			super( representation );
		}

		public AlphabetParameterSet clone() throws CloneNotSupportedException
		{
			return (AlphabetParameterSet) super.clone();
		}

		public String getInstanceName()
		{
			return getClass().getName();
		}
	}
}