/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.assessment;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.Sample.PartitionMethod;
import de.jstacs.parameters.EnumParameter;
import de.jstacs.parameters.ExpandableParameterSet;
import de.jstacs.parameters.ParameterSet;
import de.jstacs.parameters.ParameterSetContainer;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.NumberValidator;
import de.jstacs.results.CategoricalResult;
import de.jstacs.results.NumericalResult;
import de.jstacs.results.Result;

/**
 * This class implements a <code>ClassifierAssessmentAssessParameterSet</code> that must be used
 * to call method <code>assess()</code> of a <code>RepatedHoldOutExperiment</code>.
 * It contains user-specific parameters necessary for a run of a <code>RepeatedHoldOutExperiment</code>.
 * 
 * @author andr|e gohr (a0coder (nospam:@) gmail (nospam:.) com)
 *
 */
public class RepeatedHoldOutAssessParameterSet extends ClassifierAssessmentAssessParameterSet {


//	**********************
//	class variables
//	**********************

//	**********************
//	class methods
//	**********************

//	**********************
//	member variables
//	**********************

//	**********************
//	constructors
//	**********************

	
	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	protected RepeatedHoldOutAssessParameterSet
	(Class alternativeInstanceClass) {
	super(alternativeInstanceClass);
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public RepeatedHoldOutAssessParameterSet
	()throws UnsupportedOperationException {
	super();
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public RepeatedHoldOutAssessParameterSet
	(StringBuffer representation) throws NonParsableException {
	super(representation);
	}

	/**
	 * 
	 * @param dataSplitMethod	defines the method used to split user-supplied data into
	 * 							<code>k</code> mutually exclusive, random-splits
	 * 							(available options are: {@link PartitionMethod#PARTITION_BY_NUMBER_OF_ELEMENTS} and {@link PartitionMethod#PARTITION_BY_NUMBER_OF_SYMBOLS}) See docu of <code>Sample</code>
	 * 							for further details.
	 *
	 * @param elementLength 	defines the length of elements (sequences) the classifiers
	 * 							to be assessed are able to classify
	 * 
	 * @param exceptionIfMPNotComputable a <code>RepeatedHoldOutParameterSet</code> is used
	 * 							in combination with an <code>MeasureParameters</code>-object
	 * 							to call <code>assess</code>-methods of <code>RepeatedHoldOutExperiment</code>s.
	 * 						    If <code>exceptionIfMPNotComputable=true</code> then an expection is thrown
	 * 							in case of a selected measure-parameters that could not be computed.
	 * 
	 * @param repeats 			the number of repeates of each iteration 
	 * 							(mutually exclusive, randomly split data to obtain
	 * 							test- and train-data-sets, 
	 * 							train classifiers using train-data-sets
	 * 							and test them using test-data-sets)
	 * 							of that <code>RepeatedHoldOutExperiment</code> this 
	 * 							<code>RepatedHoldOutAssessParameterSet</code> is used with
	 * 
	 * @param percents 			this array containes class-wise the percentage of the user-supplied data
	 *  						that should be used as test-data in each iteration 
	 *  						of that <code>RepeatedHoldOutExperiment</code> this 
	 * 							<code>RepatedHoldOutAssessParameterSet</code> is used with
	 * 
	  * @throws IllegalValueException is thrown in case of out-of-range or invalid given parameters
	 */
	public RepeatedHoldOutAssessParameterSet
	(PartitionMethod dataSplitMethod, int elementLength, boolean exceptionIfMPNotComputable, 
	int repeats, double[] percents)
	throws IllegalValueException {
	super(elementLength, exceptionIfMPNotComputable);
	
		this.parameters.get(2).setValue( new Integer(repeats) );
		
		ParameterSet[] tempPSA = new ParameterSet[percents.length];
		for(int i=0; i<tempPSA.length;			
			tempPSA[i]=getParameterSetContainingASingleDoubleValue(percents[i++])
		);
		
		( (ExpandableParameterSet)(( (ParameterSetContainer)(this.parameters.get(3)) ).getValue()) ).replaceContentWith(tempPSA);
	
		(this.parameters.get(4)).setValue(dataSplitMethod);
		
	}
	
	

//	**********************
//	member methods
//	**********************

	/**
	 * Creates a new ParameterSet containing a single Double-SimpleParameter.
	 * This ParameterSet is used as a part of the ExpandableParameterSet that contains
	 * the test-data-percent for a specific class. <br>
	 * @param percent the double-value to be contained in the returned ParameterSet.
	 * If percet=Double.NaN, no values is contained in the returned ParameterSet 
	 * (The SimpleParameter contained in the returned ParameterSet conains no value).
	 * @throws IllegalValueException 
	 */
	private ParameterSet getParameterSetContainingASingleDoubleValue
	(double percent) 
	throws IllegalValueException{
		
		ParameterSet ret = new ParameterSet(){
   								protected void loadParameters
   								() throws Exception{
   								initParameterList(1);
   								this.parameters.add( new SimpleParameter(
   										DataType.DOUBLE,
   											"percent",
   											"Defines the percentage of the entire given data (for a " +
   											"specific class) should be used as test-data in a " +
   											"RepeatedHoldOutExperiment.",
   											true,
											new NumberValidator<Double>(0d,1d) )
   										);
   								}

   								public String getInstanceName
   								(){
   			                    return "";
   								}

   								public String getInstanceComment
   								(){
   								return "Contains only one Double-object. This double-value " +
   								"defines the percentage of the entire given data " +
   								"(for a specific class) should be used as test-data " +
   								"in a RepeatedHoldOutExperiment.";
   								}
			};
			if( !Double.isNaN( percent ) ){
				ret.getParameterAt(0).setValue(new Double(percent));
			}
	
	return ret;
	}
	
	
	@Override
	protected void initializeMyParametersArrayList
	(){
		initParameterList(5);
	}
	
	
	@Override
	protected void loadParameters
	() throws Exception {
	super.loadParameters();	
	
		//2-k
		this.parameters.add( new SimpleParameter(DataType.INT,
													"repeats",
													"Determines how often the procedure of " +
													"train/test classifers using random created " +
													"train- and test-data should be repeated.",
													true,
													new NumberValidator<Integer>(1,Integer.MAX_VALUE))
						    );
		
		//3-percents
		this.parameters.add( new ParameterSetContainer("testDataPercentage",
													   "A RepeatedHoldOutExperiment splits " +
													   "the given data (for each class) in each iteration into " +
													   "a test-part and a train-part. This ParameterSetcontainer " +
													   "contains an ExpandableParameterSet that contains for each class " +
													   "the percent of the entire data (for each class), that should be used " +
													   "as test-data. (1-percent) defines the percent of train-data.",
													   new ExpandableParameterSet(null,
															   					  getParameterSetContainingASingleDoubleValue(Double.NaN),
													   							  "percent",
													   							  "At pos i in this Expandable ParameterSet " +
													   							  "defines the percent of all given data, " +
													   							  "that should be used as test-data for class " +
													   							  "i in a RepeatedHoldOutExperiment."
													   	)//new ExpandableParameterSet
											)//new ParameterSetContainer
							);//this.parameters.add(...)
		

		//4-dataSplitMethod
		this.parameters.add( new EnumParameter(PartitionMethod.class,"The method used to compute the percentages of the partitions",true )  
							);
	}

	@Override
	public String getInstanceName() {
	return "RepeatedHoldOutExperiment";
	}

	@Override
	public String getInstanceComment() {
	return "This RepeatedHoldOutAssessParameterSet may be used to pass parameters into " +
			"RepeatedHoldOutExperiment's method assess().";
	}

	
	/**
	 * @return 	the repeates defined by this <code>RepatedHoldOutAssessParameterSet</code>
	 * 			(repeats defines how many iterations (train and test classifiers) 
	 * 			of that <code>RepeatedHoldOutExperiment</code>
	 * 			this <code>RepeatedHoldOutAssessParameterSet</code> is used with are performed)
	 */
	public int getRepeats
	(){
	return ((Integer)(this.getParameterAt(2).getValue())).intValue();
	}
	
	/**
	 * @return 	an array containing for each class the percentage
	 * 			of user supplied-data that is used in each iterations as
	 * 			test-data-set
	 */
	public double[] getPercents
	(){
		
		ExpandableParameterSet tempEPS = (ExpandableParameterSet)(this.getParameterAt(3).getValue());
		
		double[] ret = new double[tempEPS.getNumberOfParameters()];
		
		for(int i=0; i<ret.length;i++){
			//holy shit, that's realy unsexy
			ret[i]=((Double)(((ParameterSet)(tempEPS.getParameterAt(i).getValue())).getParameterAt(0).getValue())).doubleValue();
		}
		
	return ret;
	}
	
	/**
	 * @return 	The {@link de.jstacs.data.Sample.PartitionMethod} defining how the mutually exclusive, random-splits
	 * 			of user-supplied data are generated. See class <code>Sample</code> for further details.
	 */
	public PartitionMethod getDataSplitMethod
	(){
	return (PartitionMethod) ((EnumParameter) getParameterAt(4)).getValue();
	}

	@Override
	public Collection<Result> getAnnotation()
	{
		ArrayList<Result> l = new ArrayList<Result>(3);
		l.add( new NumericalResult( "repeats", "The number of iterations", getRepeats() ) );
		l.add( new CategoricalResult( "percentage", "The percentage of the entire data (for each class), that was used in an assessment",
				Arrays.toString( getPercents() ) ) );
		l.add( new CategoricalResult( "dataSplitMethod", "Describes how data should be splitted in ClassiefierAssessment.evaluateClassifier())",
				getDataSplitMethod().name() ) );
		return l;
	}
	
}
